package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.TextFormat;
import io.grpc.Status;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.ReadMetricsDataRequest;
import ru.yandex.monitoring.api.v3.ReadMetricsDataResponse;
import ru.yandex.monitoring.api.v3.WriteMetricsDataRequest;
import ru.yandex.monitoring.api.v3.WriteMetricsDataResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.exception.handlers.CommonApiExceptionHandler;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsDataService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DataReadDtoConverter;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.push.PushFormats;
import ru.yandex.solomon.gateway.push.PushMetricProcessor;
import ru.yandex.solomon.gateway.push.PushRequest;
import ru.yandex.solomon.gateway.push.PushStatusToHttpStatus;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.selfmon.trace.SpanAwareFuture;
import ru.yandex.solomon.util.time.Deadline;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class MetricsDataServiceImpl implements MetricsDataService {
    private static final Logger logger = LoggerFactory.getLogger(MetricsDataServiceImpl.class);

    private final Authorizer authorizer;
    private final DataClient dataClient;
    private final SolomonConfHolder confHolder;
    private final ProjectsDao projectsDao;
    private final PushMetricProcessor pushMetricProcessor;

    @Autowired
    public MetricsDataServiceImpl(
            Authorizer authorizer,
            DataClient dataClient,
            SolomonConfHolder confHolder,
            ProjectsDao projectsDao,
            PushMetricProcessor pushMetricProcessor)
    {
        this.authorizer = authorizer;
        this.dataClient = dataClient;
        this.confHolder = confHolder;
        this.projectsDao = projectsDao;
        this.pushMetricProcessor = pushMetricProcessor;
    }

    @Override
    public CompletableFuture<ReadMetricsDataResponse> read(ReadMetricsDataRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.DATA_READ)
                .thenCompose(account -> checkForeignRefs(request.getProjectId()))
                .thenCompose(aVoid -> doRead(request, subject));
    }

    private CompletableFuture<ReadMetricsDataResponse> doRead(ReadMetricsDataRequest request, AuthSubject subject) {
        Instant deadline = Deadline.defaultDeadline();
        String subjectId = AuthSubject.getLogin(subject, subject.getUniqueId());
        DataRequest dataRequest = DataReadDtoConverter.toModel(request, deadline, subjectId);
        return dataClient.readData(dataRequest).thenApply(DataReadDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<WriteMetricsDataResponse> write(WriteMetricsDataRequest request, AuthSubject subject) {
        if (request.getContainerCase() != WriteMetricsDataRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        if (request.getDataCase() == WriteMetricsDataRequest.DataCase.RAW_DATA) {
            WriteMetricsDataRequest.RawData rawData = request.getRawData();
            return writeRawData(
                    request.getProjectId(),
                    request.getCluster(),
                    request.getService(),
                    rawData.getContentType(),
                    rawData.getContent().toByteArray(),
                    subject);
        } else if (request.getDataCase() == WriteMetricsDataRequest.DataCase.TYPED_DATA) {
            var data = request.getTypedData();
            return writeRawData(
                    request.getProjectId(),
                    request.getCluster(),
                    request.getService(),
                    ru.yandex.monlib.metrics.encode.MetricFormat.PROTOBUF.contentType(),
                    data.toByteArray(),
                    subject);
        }
        return CompletableFuture.failedFuture(new BadRequestException("unknown data type: " + request.getDataCase()));
    }

    public CompletableFuture<WriteMetricsDataResponse> writeRawData(
            String projectId,
            String cluster,
            String service,
            @Nullable String contentType,
            byte[] body,
            AuthSubject subject) {
        return authorizer.authorize(subject, projectId, Permission.DATA_WRITE)
                .thenCompose(account -> doWriteRawData(projectId, cluster, service, contentType, body, account));
    }

    public CompletableFuture<WriteMetricsDataResponse> doWriteRawData(
            String projectId,
            String cluster,
            String service,
            @Nullable String contentType,
            byte[] body,
            Account account) {
        MetricFormat format = computeMetricsFormat(contentType);
        ShardKey shardKey = computeShardKey(projectId, cluster, service);
        ByteBuf content = Unpooled.wrappedBuffer(body);
        var request = new PushRequest(shardKey, format, content, body.length, account);
        return SpanAwareFuture.wrap(pushMetricProcessor.processPush(request, false))
                .whenComplete((r, t) -> logResult(shardKey, r, t))
                .thenCompose(r -> {
                    HttpStatus httpStatus = PushStatusToHttpStatus.convert(r.getStatus());
                    if (httpStatus == HttpStatus.OK) {
                        return CompletableFuture.completedFuture(WriteMetricsDataResponse.newBuilder()
                                .setWrittenMetricsCount(r.getSuccessMetricCount())
                                .build());
                    } else {
                        Status status = CommonApiExceptionHandler.mapToGrpcStatus(httpStatus).toStatus()
                                .withDescription(r.getErrorMessage());
                        return CompletableFuture.failedFuture(status.asRuntimeException());
                    }
                });
    }

    private static MetricFormat computeMetricsFormat(@Nullable String contentType) {
        if (StringUtils.isEmpty(contentType)) {
            throw new BadRequestException(CONTENT_TYPE + " header is missing");
        }
        MetricFormat format = PushFormats.byContentType(contentType);
        if (format == MetricFormat.METRIC_FORMAT_UNSPECIFIED) {
            String msg = "cannot determine metrics format by given content type: '" + contentType + '\'';
            throw new BadRequestException(msg);
        }
        return format;
    }

    private static ShardKey computeShardKey(String projectId, String cluster, String service) {
        final ShardKey shardKey;
        try {
            shardKey = new ShardKey(projectId, cluster, service);
        } catch (Throwable e) {
            String msg = String.format(
                    "incomplete values for (project='%s', cluster='%s', service='%s')",
                    projectId, cluster, service);
            throw new BadRequestException(msg);
        }
        return shardKey;
    }

    private CompletableFuture<Void> checkForeignRefs(String projectId) {
        SolomonConfWithContext conf = confHolder.getConf();

        Project project = conf == null ? null : conf.getProject(projectId);

        if (project != null) {
            return CompletableFuture.completedFuture(null);
        }

        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }

    private static void logResult(ShardKey shardKey, TDataProcessResponse response, @Nullable Throwable t) {
        if (t != null) {
            Throwable cause = CompletableFutures.unwrapCompletionException(t);
            if (cause instanceof UnknownShardException) {
                logger.warn("push into unknown shard {}", shardKey);
            } else {
                logger.warn("push into {} failed", shardKey, t);
            }
        } else {
            logger.info("push into {} processed: {}", shardKey, TextFormat.shortDebugString(response));
        }
    }
}
