package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.monitoring.api.v3.CancelDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.CreateDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.DeleteMetricsOperation;
import ru.yandex.monitoring.api.v3.GetDeleteMetricsOperationRequest;
import ru.yandex.monitoring.api.v3.ListDeleteMetricsOperationsRequest;
import ru.yandex.monitoring.api.v3.ListDeleteMetricsOperationsResponse;
import ru.yandex.monitoring.api.v3.Operation;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsDeletionService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DeleteMetricsOperationConverter;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationMetrics;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationStatus;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationTracker;
import ru.yandex.solomon.gateway.tasks.deleteMetrics.DeleteMetricsTaskHandler;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.scheduler.TaskScheduler;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.concurrent.TimeUnit.DAYS;
import static ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager.unpackData;
import static ru.yandex.solomon.staffOnly.annotations.ManagerMethod.ExecuteMethod.POST;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;

/**
 * @author Stanislav Kashirin
 */
@Component
@ParametersAreNonnullByDefault
public class MetricsDeletionServiceImpl implements MetricsDeletionService {

    private static final int MAX_SELECTORS_LENGTH = 100_000;
    private static final int MAX_DESCRIPTION_LENGTH = 1_000;

    private static final Logger logger = LoggerFactory.getLogger(MetricsDeletionServiceImpl.class);

    private final Authorizer authorizer;
    private final SolomonConfHolder confHolder;
    private final TaskScheduler taskScheduler;
    private final ShardsDao shardsDao;
    private final DeleteMetricsOperationManager operationManager;
    private final DeleteMetricsOperationMetrics operationMetrics;
    private final DeleteMetricsOperationTracker operationTracker;

    public MetricsDeletionServiceImpl(
        Authorizer authorizer,
        SolomonConfHolder confHolder,
        TaskScheduler taskScheduler,
        ShardsDao shardsDao,
        DeleteMetricsOperationManager operationManager,
        DeleteMetricsOperationMetrics operationMetrics,
        DeleteMetricsOperationTracker operationTracker)
    {
        this.authorizer = authorizer;
        this.confHolder = confHolder;
        this.taskScheduler = taskScheduler;
        this.shardsDao = shardsDao;
        this.operationManager = operationManager;
        this.operationMetrics = operationMetrics;
        this.operationTracker = operationTracker;
    }

    @Override
    public CompletableFuture<Operation> create(
        CreateDeleteMetricsOperationRequest request,
        AuthSubject subject)
    {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
            .thenCompose(account -> doCreate(request, account));
    }

    private CompletableFuture<Operation> doCreate(
        CreateDeleteMetricsOperationRequest request,
        Account account)
    {
        return schedule(request, account.getId(), Optional.empty())
            .thenApply(DeleteMetricsOperationConverter::toOperation);
    }

    @Override
    public CompletableFuture<DeleteMetricsOperation> get(
        GetDeleteMetricsOperationRequest request,
        AuthSubject subject)
    {
        return doGet(request.getOperationId())
            .thenCompose(operation -> {
                var projectId = operation.getProjectId();

                return authorizer.authorize(subject, projectId, Permission.CONFIGS_GET)
                    .thenApply(account -> operation);
            });
    }

    private CompletableFuture<DeleteMetricsOperation> doGet(String operationId) {
        return operationManager.getOperationOrThrow(operationId)
            .thenApply(DeleteMetricsOperationConverter::toProto);
    }

    @Override
    public CompletableFuture<Operation> cancel(
        CancelDeleteMetricsOperationRequest request,
        AuthSubject subject)
    {
        return doCancel(request, subject)
            .thenApply(DeleteMetricsOperationConverter::toOperation);
    }

    @Override
    public CompletableFuture<DeleteMetricsOperation> cancelV2(
        CancelDeleteMetricsOperationRequest request,
        AuthSubject subject)
    {
        return doCancel(request, subject)
            .thenApply(DeleteMetricsOperationConverter::toProto);
    }

    private CompletableFuture<LongRunningOperation> doCancel(
        CancelDeleteMetricsOperationRequest request,
        AuthSubject subject)
    {
        return operationManager.getOperationOrThrow(request.getOperationId())
            .thenCompose(
                operation -> authorizer.authorize(subject, operation.containerId(), Permission.CONFIGS_CREATE)
                    .thenCompose(account -> rollback(operation, account.getId())));
    }

    @Override
    public CompletableFuture<ListDeleteMetricsOperationsResponse> list(
        ListDeleteMetricsOperationsRequest request,
        AuthSubject subject)
    {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
            .thenCompose(account -> doList(request));
    }

    private CompletableFuture<ListDeleteMetricsOperationsResponse> doList(
        ListDeleteMetricsOperationsRequest request)
    {
        var pageOpts = new TokenPageOptions(
            (int) request.getPageSize(),
            request.getPageToken());

        return operationManager.listOperations(request.getProjectId(), pageOpts)
            .thenApply(page -> {
                var rs = page.map(DeleteMetricsOperationConverter::toProto);

                return ListDeleteMetricsOperationsResponse.newBuilder()
                    .addAllOperations(rs.getItems())
                    .setNextPageToken(rs.getNextPageToken())
                    .build();
            });
    }

    @ManagerMethod(executeMethod = POST)
    CompletableFuture<String> deleteMetrics(
        @ManagerMethodArgument(name = "projectId") String projectId,
        @ManagerMethodArgument(name = "selectors") String rawSelectors)
    {
        return deleteMetrics(projectId, rawSelectors, "");
    }

    @ManagerMethod(executeMethod = POST)
    CompletableFuture<String> deleteMetrics(
        @ManagerMethodArgument(name = "projectId") String projectId,
        @ManagerMethodArgument(name = "selectors") String rawSelectors,
        @ManagerMethodArgument(name = "permanentDeletionDelay") String permanentDeletionDelay)
    {
        var request = CreateDeleteMetricsOperationRequest.newBuilder()
            .setProjectId(projectId)
            .setSelectors(rawSelectors)
            .setDescription("invoked from staff only")
            .build();
        var delay = DurationUtils.parseDuration(permanentDeletionDelay);

        return schedule(request, "", delay)
            .thenApply(LongRunningOperation::operationId);
    }

    @ManagerMethod(executeMethod = POST)
    CompletableFuture<LongRunningOperation> rollbackOperation(
        @ManagerMethodArgument(name = "operationId") String operationId)
    {
        return operationManager.getOperationOrThrow(operationId)
            .thenCompose(operation -> rollback(operation, ""));
    }

    private CompletableFuture<LongRunningOperation> schedule(
        CreateDeleteMetricsOperationRequest request,
        String login,
        Optional<Duration> permanentDeletionDelay)
    {
        var validationStatus = validate(request);
        if (!validationStatus.isOk()) {
            return failedFuture(validationStatus.asRuntimeException());
        }

        var selectors = Selectors.parse(request.getSelectors())
            .toBuilder()
            .addOverride(LabelKeys.PROJECT, request.getProjectId())
            .build();

        var shardKey = ShardSelectors.getShardKeyOrNull(selectors);
        if (shardKey == null) {
            return failedFuture(
                Status.INVALID_ARGUMENT
                    .withDescription(
                        "only single shard deletions are supported,"
                            + " ensure that 'cluster' and 'service' labels contain exact value each")
                    .asRuntimeException());
        }

        return findShard(shardKey)
            .thenCompose(shard -> schedule(request, login, permanentDeletionDelay, selectors, shard));
    }

    private CompletableFuture<LongRunningOperation> schedule(
        CreateDeleteMetricsOperationRequest request,
        String login,
        Optional<Duration> permanentDeletionDelay,
        Selectors selectors,
        Shard shard)
    {
        var now = System.currentTimeMillis();
        var permanentDeletionDelayMillis = permanentDeletionDelay
            .map(Duration::toMillis)
            .orElseGet(() -> DAYS.toMillis(30) + randomize(DAYS.toMillis(1)));

        var params = DeleteMetricsParams.newBuilder()
            .setProjectId(request.getProjectId())
            .setSelectors(SelectorsFormat.format(selectors))
            .addNumIds(shard.getNumId())
            .setDescription(request.getDescription())
            .setCreatedBy(login)
            .setCreatedAt(now)
            .setPermanentDeletionAt(now + permanentDeletionDelayMillis)
            .build();

        var task = DeleteMetricsTaskHandler.task(params);
        var operationId = task.id();

        return operationManager.checkProjectLimitReached(request.getProjectId(), now)
            .thenCompose(reached -> {
                if (reached) {
                    return failedFuture(
                        Status.RESOURCE_EXHAUSTED
                            .withDescription(
                                "Daily operation limit ("
                                    + DeleteMetricsOperationManager.PROJECT_LIMIT
                                    + ") has been reached for project "
                                    + request.getProjectId()
                                    + ". Please, try again later.")
                            .asRuntimeException());
                }

                return taskScheduler.schedule(task)
                    .thenCompose(scheduled -> {
                        operationTracker.operationStarted(
                            now,
                            login,
                            ContainerType.PROJECT,
                            request.getProjectId(),
                            operationId,
                            request.getSelectors());
                        operationMetrics.operationStarted(request.getProjectId());
                        operationMetrics.operationStarted(request.getProjectId(), shard.getId());

                        return tryCreateOperation(operationId, params);
                    });
            });
    }

    private CompletableFuture<LongRunningOperation> tryCreateOperation(
        String operationId,
        DeleteMetricsParams params)
    {
        return operationManager.tryCreateOperation(operationId, params)
            .thenApply(result -> {
                if (result.isOk()) {
                    return result.operation();
                }

                if (result.isAlreadyExists()) {
                    logger.info("delete metrics operation already created: {}", operationId);
                    return result.operation();
                }

                logger.error(
                    "failed to create delete metrics operation, will be created later: {} {}",
                    operationId,
                    result.status());
                return result.operation();
            });
    }

    private CompletableFuture<LongRunningOperation> rollback(LongRunningOperation operation, String login) {
        var conf = confHolder.getConfOrThrow();

        var now = System.currentTimeMillis();
        var rescheduleFuture = taskScheduler.reschedule(
            operation.operationId(),
            now,
            DeleteMetricsTaskHandler.newRollbackOperator(conf, login, now));

        return rescheduleFuture.thenCompose(success -> {
            if (!success) {
                return completedFuture(operation);
            }

            var data = unpackData(operation).toBuilder()
                .setProgressPercentage(0)
                .setCancelledAt(now)
                .setCancelledBy(login)
                .build();
            var update = operation.toBuilder()
                .setUpdatedAt(now)
                .setStatus(DeleteMetricsOperationStatus.CANCELLING.value)
                .setData(Any.pack(data))
                .build();

            operationTracker.operationCancelled(
                now,
                login,
                ContainerType.PROJECT,
                operation.containerId(),
                operation.operationId());
            operationMetrics.operationCancelled(operation.containerId());
            for (int i = 0; i < data.getNumIdsCount(); i++) {
                var shard = conf.getShardByNumIdOrNull(data.getNumIds(i));
                if (shard != null) {
                    operationMetrics.operationCancelled(operation.containerId(), shard.getId());
                }
            }

            return operationManager.forceUpdateOperation(update)
                .thenApply(i -> update);
        }).whenComplete((i, t) -> {
            if (t != null) {
                operationMetrics.operationCancelFailed(operation.containerId());
            }
        });
    }

    private CompletableFuture<Shard> findShard(ShardKey shardKey) {
        return shardsDao.findByShardKey(shardKey.getProject(), shardKey.getCluster(), shardKey.getService())
            .thenApply(shard -> shard.orElseThrow(() -> shardNotFound(shardKey)));
    }

    private static StatusRuntimeException shardNotFound(ShardKey shardKey) {
        return Status.NOT_FOUND
            .withDescription("shard not found: " + shardKey)
            .asRuntimeException();
    }

    private static Status validate(CreateDeleteMetricsOperationRequest request) {
        if (request.getSelectors().length() > MAX_SELECTORS_LENGTH) {
            return Status.INVALID_ARGUMENT
                .withDescription(
                    "too long selectors: " + request.getSelectors().length() + " > " + MAX_SELECTORS_LENGTH);
        }
        if (request.getDescription().length() > MAX_DESCRIPTION_LENGTH) {
            return Status.INVALID_ARGUMENT
                .withDescription(
                    "too long description: " + request.getDescription().length() + " > " + MAX_DESCRIPTION_LENGTH);
        }
        return Status.OK;
    }
}
