package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.ListAllLabelValuesRequest;
import ru.yandex.monitoring.api.v3.ListAllLabelValuesResponse;
import ru.yandex.monitoring.api.v3.ListLabelKeysRequest;
import ru.yandex.monitoring.api.v3.ListLabelKeysResponse;
import ru.yandex.monitoring.api.v3.ListLabelValuesRequest;
import ru.yandex.monitoring.api.v3.ListLabelValuesResponse;
import ru.yandex.monitoring.api.v3.ListMetricNamesRequest;
import ru.yandex.monitoring.api.v3.ListMetricNamesResponse;
import ru.yandex.monitoring.api.v3.ListMetricsRequest;
import ru.yandex.monitoring.api.v3.ListMetricsResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.v2.managers.MetricsManager;
import ru.yandex.solomon.gateway.api.v2.managers.ShardLabelsManager;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsMetaService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.MetricsMetaDtoConverter;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class MetricsMetaServiceImpl implements MetricsMetaService {
    private final Authorizer authorizer;
    private final MetricsManager metricsManager;
    private final ShardLabelsManager shardLabelsManager;

    @Autowired
    public MetricsMetaServiceImpl(Authorizer authorizer, MetricsManager metricsManager, ShardLabelsManager shardLabelsManager) {
        this.authorizer = authorizer;
        this.metricsManager = metricsManager;
        this.shardLabelsManager = shardLabelsManager;
    }

    @Override
    public CompletableFuture<ListMetricsResponse> listMetrics(ListMetricsRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListMetricsRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.METRICS_GET)
                .thenCompose(account -> doListMetrics(request));
    }

    private CompletableFuture<ListMetricsResponse> doListMetrics(ListMetricsRequest request) {
        Selectors selectors = parseSelectors(request.getProjectId(), request.getSelectors());
        TokenPageOptions pageOptions = new TokenPageOptions((int) request.getPageSize(), request.getPageToken());
        boolean useNewFormat = !request.getUseOldFormat();

        return metricsManager.findMetrics(selectors, request.getForceZoneId(), pageOptions, useNewFormat)
                .thenApply(MetricsMetaDtoConverter::toListMetricsResponse);
    }

    @Override
    public CompletableFuture<ListMetricNamesResponse> listMetricNames(
            ListMetricNamesRequest request,
            AuthSubject subject)
    {
        if (request.getContainerCase() != ListMetricNamesRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.METRIC_NAMES_GET)
                .thenCompose(account -> doListMetricNames(request));
    }

    private CompletableFuture<ListMetricNamesResponse> doListMetricNames(ListMetricNamesRequest request) {
        Instant deadline = Deadline.defaultDeadline();
        Selectors selectors = parseSelectors(request.getProjectId(), request.getSelectors());
        int limit = pageSizeToLimit((int) request.getPageSize());
        return metricsManager.findMetricNames(selectors, request.getNameFilter(), LabelValidationFilter.ALL, limit, request.getForceZoneId(), deadline)
                .thenApply(MetricsMetaDtoConverter::toListMetricNamesResponse);
    }

    @Override
    public CompletableFuture<ListLabelKeysResponse> listLabelKeys(ListLabelKeysRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListLabelKeysRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        Instant deadline = Deadline.defaultDeadline();
        Selectors selectors = parseSelectors(request.getProjectId(), request.getSelectors());
        boolean useNewFormat = !request.getUseOldFormat();

        return metricsManager.findLabelNames(selectors, useNewFormat, request.getForceZoneId(), deadline)
                .thenApply(MetricsMetaDtoConverter::toListLabelKeysResponse);
    }

    @Override
    public CompletableFuture<ListLabelValuesResponse> listLabelValues(ListLabelValuesRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListLabelValuesRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.METRIC_LABELS_GET)
                .thenCompose(account -> doListLabelValues(request));
    }

    private CompletableFuture<ListLabelValuesResponse> doListLabelValues(ListLabelValuesRequest request) {
        String labelKey = request.getLabelKey();
        if (labelKey.equals(LabelKeys.PROJECT)) {
            return CompletableFuture.failedFuture(notAllowedProjectLabelKey());
        }

        Instant deadline = Deadline.defaultDeadline();
        Selectors selectors = parseSelectors(request.getProjectId(), request.getSelectors());
        boolean useNewFormat = !request.getUseOldFormat();
        int limit = pageSizeToLimit((int) request.getPageSize());

        if (mustShowClusterServiceOnly(selectors, labelKey)) {
            LabelValuesResponse response = shardLabelsManager.findShardLabelValues(
                    List.of(labelKey),
                    selectors,
                    request.getValueFilter(),
                    LabelValidationFilter.ALL,
                    limit);

            return CompletableFuture.completedFuture(MetricsMetaDtoConverter.toListLabelValuesResponse(response, true));
        }

        return metricsManager.findLabelValues(
                List.of(labelKey),
                selectors,
                request.getValueFilter(),
                LabelValidationFilter.ALL,
                limit,
                useNewFormat,
                request.getForceZoneId(),
                deadline)
                .thenApply((response) -> MetricsMetaDtoConverter.toListLabelValuesResponse(response, false));
    }

    @Override
    public CompletableFuture<ListAllLabelValuesResponse> listAllLabelValues(
            ListAllLabelValuesRequest request,
            AuthSubject subject)
    {
        if (request.getContainerCase() != ListAllLabelValuesRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.METRIC_LABELS_GET)
                .thenCompose(account -> doListAllLabelValues(request));
    }

    private CompletableFuture<ListAllLabelValuesResponse> doListAllLabelValues(ListAllLabelValuesRequest request) {
        Instant deadline = Deadline.defaultDeadline();
        Selectors selectors = parseSelectors(request.getProjectId(), request.getSelectors());
        boolean useNewFormat = !request.getUseOldFormat();
        int limit = pageSizeToLimit((int) request.getPageSize());

        if (mustShowClusterServiceOnly(selectors, "")) {
            LabelValuesResponse response = shardLabelsManager.findShardLabelValues(
                    List.of(),
                    selectors,
                    request.getValueFilter(),
                    LabelValidationFilter.ALL,
                    limit);
            return CompletableFuture.completedFuture(MetricsMetaDtoConverter.toListAllLabelValuesResponse(selectors, response, true));
        }

        return metricsManager.findLabelValues(
                List.of(),
                selectors,
                request.getValueFilter(),
                LabelValidationFilter.ALL,
                limit,
                useNewFormat,
                request.getForceZoneId(),
                deadline)
                .thenApply(response -> MetricsMetaDtoConverter.toListAllLabelValuesResponse(selectors, response, false));
    }

    private static Selectors parseSelectors(String projectId, String selectors) {
        try {
            return Selectors.parse(selectors)
                    .toBuilder()
                    .addOverride(Selector.exact(LabelKeys.PROJECT, projectId))
                    .build();
        } catch (Throwable e) {
            throw new BadRequestException(String.format("failed to parse selectors %s", selectors));
        }
    }

    private static int pageSizeToLimit(int pageSize) {
        if (pageSize <= 0) {
            return 100;
        }
        return Math.min(pageSize, 1000);
    }

    private static BadRequestException notAllowedProjectLabelKey() {
        return new BadRequestException("\"project\" isn't allowed label key for listing");
    }

    private static boolean mustShowClusterServiceOnly(Selectors selectors, String labelKey) {
        if (!labelKey.isEmpty() && !LabelKeys.isShardKeyPart(labelKey)) {
            return false;
        }

        if (!selectors.getNameSelector().isEmpty()) {
            return false;
        }

        boolean hasCluster = selectors.hasKey(LabelKeys.CLUSTER);
        boolean hasService = selectors.hasKey(LabelKeys.SERVICE);

        if (hasCluster && hasService) {
            return false;
        }

        return selectors.stream().allMatch(s -> LabelKeys.isShardKeyPart(s.getKey()));
    }
}
