package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;

import ru.yandex.idm.IdmClient;
import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.ProjectRoleOperation;
import ru.yandex.monitoring.api.v3.ProjectRoleOperation.OperationType;
import ru.yandex.monitoring.api.v3.ProjectRoleOperationStatus;
import ru.yandex.monitoring.api.v3.ProjectRoles;
import ru.yandex.monitoring.api.v3.Role;
import ru.yandex.monitoring.api.v3.RoleListRequest;
import ru.yandex.monitoring.api.v3.RoleListResponse;
import ru.yandex.monitoring.api.v3.UidType;
import ru.yandex.monitoring.api.v3.UpdateRoleListRequest;
import ru.yandex.monitoring.api.v3.UpdateRoleListStatusResponse;
import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.gateway.api.v3.intranet.ProjectRoleService;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.staff.StaffClient;
import ru.yandex.staff.StaffGroup;
import ru.yandex.staff.UserInfo;

import static ru.yandex.monitoring.api.v3.ProjectRoleOperationStatus.OperationStatus;
import static ru.yandex.monitoring.api.v3.ProjectRoleOperationStatus.newBuilder;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectRoleServiceImpl implements ProjectRoleService {

    private final ProjectAclEntryDao projectAclEntryDao;
    private final IdmClient idmClient;
    private final TIdmConfig config;
    private final StaffClient staffClient;
    private final Authorizer authorizer;
    private final SolomonConfHolder confHolder;
    private final ProjectsDao projectsDao;

    public ProjectRoleServiceImpl(
            ProjectAclEntryDao projectAclEntryDao,
            IdmClient idmClient,
            TIdmConfig config,
            StaffClient staffClient,
            Authorizer authorizer,
            SolomonConfHolder confHolder,
            ProjectsDao projectsDao)
    {
        this.projectAclEntryDao = projectAclEntryDao;
        this.idmClient = idmClient;
        this.config = config;
        this.staffClient = staffClient;
        this.authorizer = authorizer;
        this.confHolder = confHolder;
        this.projectsDao = projectsDao;
    }

    @Override
    public CompletableFuture<RoleListResponse> list(RoleListRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> projectAclEntryDao.list(request.getProjectId(), (int) request.getPageSize(), request.getPageToken())
                        .thenCompose(projectAclEntries -> resolveGroups(projectAclEntries)
                                .thenCompose(mapGroups -> resolveUsers(projectAclEntries)
                                        .thenCompose(usersMap -> {
                                            var roles = projectAclEntries.getItems().stream()
                                                    .map(entry -> toRoleDto(entry, mapGroups, usersMap))
                                                    .collect(Collectors.toList());
                                            if (roles.isEmpty()) {
                                                // return owner anyway
                                                return ownerRolesFallback(request.getProjectId());
                                            }
                                            return CompletableFuture.completedFuture(RoleListResponse.newBuilder()
                                                    .addAllProjectRoles(roles)
                                                    .setNextPageToken(projectAclEntries.getNextPageToken())
                                                    .build());
                                        })
                                )));
    }

    @Override
    public CompletableFuture<UpdateRoleListStatusResponse> status(UpdateRoleListRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.ROLES_UPDATE)
                .thenCompose(account -> {
                    List<CompletableFuture<ProjectRoleOperationStatus>> futures = new ArrayList<>(request.getProjectRoleOperationsCount());
                    for (var operation : request.getProjectRoleOperationsList()) {
                        futures.add(checkOperation(request, operation));
                    }
                    return CompletableFutures.allOf(futures)
                            .thenApply(projectRoleOperationStatuses -> UpdateRoleListStatusResponse.newBuilder()
                                    .setProjectId(request.getProjectId())
                                    .addAllProjectRoleOperationStatuses(projectRoleOperationStatuses)
                                    .build());
                });
    }

    private CompletableFuture<ProjectRoleOperationStatus> checkOperation(UpdateRoleListRequest request, ProjectRoleOperation operation) {
        return projectAclEntryDao.find(request.getProjectId(), operation.getUid(), mapType(operation.getUidType()))
                .thenApply(projectAclEntryOptional -> {
                    OperationStatus status;
                    if (operation.getOperationType() == OperationType.ADD) {
                        status = projectAclEntryOptional.isPresent() && projectAclEntryOptional.get().getRoles().contains(mapRole(operation.getRole()))
                                ? OperationStatus.COMPLETED
                                : OperationStatus.NOT_COMPLETED;
                    } else if (operation.getOperationType() == OperationType.DELETE) {
                        if (projectAclEntryOptional.isPresent()) {
                            status = !projectAclEntryOptional.get().getRoles().contains(mapRole(operation.getRole()))
                                    ? OperationStatus.COMPLETED
                                    : OperationStatus.NOT_COMPLETED;
                        } else {
                            status = OperationStatus.COMPLETED;
                        }
                    } else {
                        throw new IllegalArgumentException("Unspecified operation type " + operation.getOperationType());
                    }
                    return newBuilder()
                            .setProjectRoleOperation(operation)
                            .setOperationStatus(status)
                            .build();
                });
    }

    @Override
    public CompletableFuture<Empty> update(UpdateRoleListRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.ROLES_UPDATE)
                .thenCompose(account -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(request.getProjectRoleOperationsCount());
                    for (var operation : request.getProjectRoleOperationsList()) {
                        if (OperationType.ADD == operation.getOperationType()) {
                            var requestDto = newRequest(request.getProjectId(), operation);
                            futures.add(idmClient.requestRole(requestDto));
                        } else if (OperationType.DELETE == operation.getOperationType()) {
                            var requestDto = newRequest(request.getProjectId(), operation);
                            futures.add(idmClient.deleteRole(requestDto));
                        } else {
                            throw new IllegalArgumentException("Unspecified operation type " + operation.getOperationType());
                        }
                    }
                    return CompletableFutures.allOf(futures)
                            .thenApply(unused -> Empty.getDefaultInstance());
                });
    }

    private RoleRequestDto newRequest(String projectId, ProjectRoleOperation operation) {
        if (operation.getUidType() == UidType.UID_TYPE_TVM || operation.getUidType() == UidType.UID_TYPE_USER) {
            return RoleRequestDto.newProjectRoleRequest(
                    operation.getUid(), projectId, mapRole(operation.getRole()), config.getSystemName(), operation.getUidType() == UidType.UID_TYPE_TVM);
        } else {
            return RoleRequestDto.newProjectRoleRequest(
                    Integer.parseInt(operation.getUid()), projectId, mapRole(operation.getRole()), config.getSystemName());
        }
    }

    private String mapRole(Role role) {
        if (role == Role.ROLE_ADMIN) {
            return ru.yandex.solomon.auth.roles.Role.PROJECT_ADMIN.name();
        } else if (role == Role.ROLE_EDITOR) {
            return ru.yandex.solomon.auth.roles.Role.EDITOR.name();
        } else if (role == Role.ROLE_JNS_SENDER) {
            return ru.yandex.solomon.auth.roles.Role.JNS_SENDER.name();
        } else if (role == Role.ROLE_PUSHER) {
            return ru.yandex.solomon.auth.roles.Role.PUSHER.name();
        } else if (role == Role.ROLE_VIEWER) {
            return ru.yandex.solomon.auth.roles.Role.VIEWER.name();
        } else if (role == Role.ROLE_MUTER) {
            return ru.yandex.solomon.auth.roles.Role.MUTER.name();
        }
        throw new IllegalArgumentException("Unspecified role " + role);
    }

    private ProjectRoles toRoleDto(ProjectAclEntry entry, Map<String, StaffGroup> groups, Map<String, UserInfo> usersMap) {
        String name = entry.getUid();
        if (entry.getType() == AclUidType.GROUP) {
            var group = groups.get(entry.getUid());
            name = group == null ? name : group.name;
        } else if (entry.getType() == AclUidType.USER) {
            var userInfo = usersMap.get(entry.getUid());
            name = userInfo == null ? name : userInfo.getFirstName() + " " + userInfo.getLastName();
        }
        return ProjectRoles.newBuilder()
                .setUid(entry.getUid())
                .setUidType(mapType(entry, groups))
                .setName(name)
                .addAllRole(entry.getRoles().stream()
                        .map(ru.yandex.solomon.auth.roles.Role::fromString)
                        .filter(Objects::nonNull)
                        .map(this::mapRole)
                        .collect(Collectors.toList()))
                .build();
    }

    private Role mapRole(ru.yandex.solomon.auth.roles.Role role) {
        if (role == ru.yandex.solomon.auth.roles.Role.PROJECT_ADMIN) {
            return Role.ROLE_ADMIN;
        } else if (role == ru.yandex.solomon.auth.roles.Role.EDITOR) {
            return Role.ROLE_EDITOR;
        }  else if (role == ru.yandex.solomon.auth.roles.Role.JNS_SENDER) {
            return Role.ROLE_JNS_SENDER;
        } else if (role == ru.yandex.solomon.auth.roles.Role.PUSHER) {
            return Role.ROLE_PUSHER;
        } else if (role == ru.yandex.solomon.auth.roles.Role.VIEWER) {
            return Role.ROLE_VIEWER;
        } else if (role == ru.yandex.solomon.auth.roles.Role.MUTER) {
            return Role.ROLE_MUTER;
        }
        throw new IllegalArgumentException("Unspecified role " + role);
    }

    private UidType mapType(ProjectAclEntry entry, Map<String, StaffGroup> groupMap) {
        if (entry.getType() == AclUidType.GROUP) {
            var group = groupMap.get(entry.getUid());
            return group != null && group.isService() ? UidType.UID_TYPE_SERVICE : UidType.UID_TYPE_GROUP;
        } else if (entry.getType() == AclUidType.USER) {
            return UidType.UID_TYPE_USER;
        } else {
            return UidType.UID_TYPE_TVM;
        }
    }

    private AclUidType mapType(UidType type) {
        if (type == UidType.UID_TYPE_GROUP || type == UidType.UID_TYPE_SERVICE) {
            return AclUidType.GROUP;
        } else if (type == UidType.UID_TYPE_USER) {
            return AclUidType.USER;
        } else {
            return AclUidType.TVM;
        }
    }

    private CompletableFuture<Map<String, StaffGroup>> resolveGroups(TokenBasePage<ProjectAclEntry> projectAclEntries) {
        var groupIds = projectAclEntries.getItems().stream()
                .filter(entry -> entry.getType() == AclUidType.GROUP)
                .map(ProjectAclEntry::getUid)
                .collect(Collectors.toList());
        return staffClient.getStaffGroup(groupIds)
                .thenApply(staffGroups -> staffGroups.stream()
                        .collect(Collectors.toMap(staffGroup -> String.valueOf(staffGroup.id), Function.identity())));
    }

    private CompletableFuture<Map<String, UserInfo>> resolveUsers(TokenBasePage<ProjectAclEntry> projectAclEntries) {
        var userIds = projectAclEntries.getItems().stream()
                .filter(entry -> entry.getType() == AclUidType.USER)
                .map(ProjectAclEntry::getUid)
                .collect(Collectors.toList());
        return staffClient.getUserInfo(userIds)
                .thenApply(userInfos -> userInfos.stream()
                        .collect(Collectors.toMap(usrInfo -> String.valueOf(usrInfo.getLogin()), Function.identity())));
    }


    private CompletableFuture<RoleListResponse> ownerRolesFallback(String projectId) {
        return getProjectById(projectId)
                .thenCompose(projectOptional -> {
                    var owner = projectOptional.map(Project::getOwner).orElse("");
                    if (owner.isEmpty()) {
                        return CompletableFuture.completedFuture(RoleListResponse.newBuilder().build());
                    }
                    if (owner.startsWith("tvm-")) {
                        var role = ProjectRoles.newBuilder()
                                .setUid(owner)
                                .setUidType(UidType.UID_TYPE_TVM)
                                .setName(owner)
                                .addRole(Role.ROLE_ADMIN)
                                .build();
                        return CompletableFuture.completedFuture(RoleListResponse.newBuilder()
                                .addProjectRoles(role)
                                .build());
                    }
                    return staffClient.getUserInfo(owner)
                            .handle((userInfo, throwable) -> {
                                var name = userInfo != null
                                        ? userInfo.getFirstName() + " " + userInfo.getLastName()
                                        : owner;
                                var role = ProjectRoles.newBuilder()
                                        .setUid(owner)
                                        .setUidType(UidType.UID_TYPE_USER)
                                        .setName(name)
                                        .addRole(Role.ROLE_ADMIN)
                                        .build();
                                return RoleListResponse.newBuilder()
                                        .addProjectRoles(role)
                                        .build();
                            });
                });
    }

    private CompletableFuture<Optional<Project>> getProjectById(String id) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return projectsDao.findById(id);
        }
        var project = conf.getProject(id);
        if (project == null) {
            return projectsDao.findById(id);
        }
        return CompletableFuture.completedFuture(Optional.ofNullable(project));
    }
}
