package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.GetProjectRequest;
import ru.yandex.monitoring.api.v3.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.ListProjectsResponse;
import ru.yandex.monitoring.api.v3.Project;
import ru.yandex.monitoring.api.v3.UpdateProjectRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayMigrationConfig;
import ru.yandex.solomon.core.conf.ProjectsManager;
import ru.yandex.solomon.core.exceptions.MethodNotAllowedException;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.api.v3.intranet.ProjectService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ProjectDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.validators.ProjectValidator;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ProjectServiceImpl implements ProjectService {
    private final Authorizer authorizer;
    private final ProjectsManager projectsManager;
    private final TGatewayCloudConfig cloudConfig;
    private final TGatewayMigrationConfig migrationConfig;
    private final FeatureFlagsHolder featureFlagsHolder;

    @Autowired
    public ProjectServiceImpl(
            Authorizer authorizer,
            ProjectsManager projectsManager,
            TGatewayConfig gatewayConfig,
            FeatureFlagsHolder featureFlagsHolder)
    {
        this.authorizer = authorizer;
        this.projectsManager = projectsManager;
        this.cloudConfig = gatewayConfig.getCloudConfig();
        this.migrationConfig = gatewayConfig.getGatewayMigrationConfig();
        this.featureFlagsHolder = featureFlagsHolder;
    }

    @Override
    public CompletableFuture<Pair<Project, Integer>> get(GetProjectRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<Project, Integer>> doGet(GetProjectRequest request) {
        ProjectValidator.validate(request);
        return projectsManager.getProject(request.getProjectId())
                .thenApply(ProjectDtoConverter::fromModelWithVersion);
    }

    @Override
    public CompletableFuture<ListProjectsResponse> list(ListProjectsRequest request, AuthSubject subject) {
        ProjectValidator.validate(request);
        return projectsManager.findV3(
                request.getFilter(),
                (int) request.getPageSize(),
                request.getPageToken())
                .thenApply(response -> {
                    var pagedResult = response.map(ProjectDtoConverter::fromModel);
                    return ListProjectsResponse.newBuilder()
                            .addAllProjects(pagedResult.getItems())
                            .setNextPageToken(pagedResult.getNextPageToken())
                            .build();
                });
    }

    @Override
    public CompletableFuture<Project> create(CreateProjectRequest request, AuthSubject subject) {
        Instant now = Instant.now();

        if (cloudConfig.getForbidProjectCreation()) {
            return CompletableFuture.failedFuture(new MethodNotAllowedException(
                    "Manual project creation is forbidden. " +
                            "Please create cloud in resource manager and wait for synchronization."));
        }

        ProjectValidator.validate(request);

        var project = ProjectDtoConverter.toModel(request, subject.getUniqueId(), now);
        project = updateCreatingProjectUsingMigrationConfig(project);
        return projectsManager.createProject(project)
                .thenApply(ProjectDtoConverter::fromModel);
    }

    private ru.yandex.solomon.core.db.model.Project updateCreatingProjectUsingMigrationConfig(ru.yandex.solomon.core.db.model.Project project) {
        var builder = project.toBuilder();

        builder.setOnlyAuthPush(true);

        if (migrationConfig.getOnlyMetricNameShardsInNewProjects()) {
            builder.setOnlyMetricNameShards(true);
        }

        return builder.build();
    }

    @Override
    public CompletableFuture<Project> update(UpdateProjectRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> {
                    boolean canUpdateAny = account.can(Permission.CONFIGS_UPDATE_ANY);
                    return doUpdate(request, canUpdateAny, subject.getUniqueId(), etag);
                });
    }

    private CompletableFuture<Project> doUpdate(UpdateProjectRequest request, boolean canUpdateAny, String login, int etag) {
        Instant now = Instant.now();
        ProjectValidator.validate(request);
        var project = ProjectDtoConverter.toModel(request, login, now, etag);
        return projectsManager.updateProject(login, project, canUpdateAny, false)
                .thenApply(ProjectDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteProjectRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> {
                    // for cloud accounts we check permissions in AccessService instead of
                    // relying on permissions stored in project
                    boolean canDeleteByNonOwner =
                            account.getAuthType() == AuthType.IAM ||
                                    account.can(Permission.CONFIGS_UPDATE_ANY) ||
                                    account.can(Permission.PROJECTS_DELETE);
                    if (!account.can(Permission.PROJECTS_DELETE)) {
                        if (!SolomonTeam.isMember(subject)) {
                            if (featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_JUGGLER_INTEGRATION, FeatureFlag.USE_PM_JUGGLER_INTEGRATION.name())) {
                                return CompletableFuture.failedFuture(new AuthorizationException("Can't delete project '" + request.getProjectId() + "' use ticket for that."));
                            }
                        }
                    }
                    return doDelete(request, account.getId(), canDeleteByNonOwner, SolomonTeam.isMember(subject));
                });
    }

    private CompletableFuture<Empty> doDelete(DeleteProjectRequest request, String login, boolean canDeleteByNonOwner, boolean skipValidation) {
        ProjectValidator.validate(request);
        return projectsManager.deleteProject(login, request.getProjectId(), canDeleteByNonOwner, skipValidation)
                .thenApply(aVoid -> Empty.getDefaultInstance());
    }
}
