package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateQuickLinksRequest;
import ru.yandex.monitoring.api.v3.DeleteQuickLinksRequest;
import ru.yandex.monitoring.api.v3.GetQuickLinksRequest;
import ru.yandex.monitoring.api.v3.QuickLinks;
import ru.yandex.monitoring.api.v3.UpdateQuickLinksRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.conf.db3.QuickLinksDao;
import ru.yandex.solomon.conf.db3.QuickLinksRecord;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.v3.intranet.QuickLinksService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.QuickLinksDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.validators.QuickLinksValidator;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class QuickLinksServiceImpl implements QuickLinksService {
    private final Authorizer authorizer;
    private final ProjectsDao projectsDao;
    private final QuickLinksDao quickLinksDao;

    @Autowired
    public QuickLinksServiceImpl(Authorizer authorizer, ProjectsDao projectsDao, QuickLinksDao quickLinksDao) {
        this.authorizer = authorizer;
        this.projectsDao = projectsDao;
        this.quickLinksDao = quickLinksDao;
    }

    @Override
    public CompletableFuture<Pair<QuickLinks, Integer>> get(GetQuickLinksRequest request, AuthSubject subject) {
        if (request.getContainerCase() != GetQuickLinksRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> quickLinksDao.read(request.getProjectId())
                        .thenApply(quickLinksOpt -> {
                            if (quickLinksOpt.isEmpty()) {
                                throw quickLinksNotFound(request.getProjectId());
                            }
                            QuickLinksRecord quickLinks = quickLinksOpt.get();
                            QuickLinks decode = QuickLinksDtoConverter.decode(quickLinks);
                            return Pair.of(decode, quickLinks.getVersion());
                        }));
    }

    @Override
    public CompletableFuture<QuickLinks> create(CreateQuickLinksRequest request, AuthSubject subject) {
        if (request.getContainerCase() != CreateQuickLinksRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(account -> {
                    Timestamp now = Timestamps.fromMillis(Instant.now().toEpochMilli());

                    QuickLinksValidator.validate(request);

                    String projectId = request.getProjectId();

                    var proto = QuickLinks.newBuilder()
                            .setProjectId(projectId)
                            .setData(request.getData())
                            .setModifiedAt(now)
                            .setModifiedBy(account.getId())
                            .build();

                    var model = QuickLinksDtoConverter.encode(proto, 0);

                    return checkForeignRefs(projectId)
                            .thenCompose(aVoid -> quickLinksDao.upsert(model))
                            .thenApply(upsertedOpt -> {
                                if (upsertedOpt.isEmpty()) {
                                    throw new ConflictException(String.format("quick links for project %s with version %s is out of date", projectId, model.getVersion()));
                                }
                                return QuickLinksDtoConverter.decode(upsertedOpt.get());
                            });
                });
    }

    @Override
    public CompletableFuture<QuickLinks> update(UpdateQuickLinksRequest request, AuthSubject subject, int etag) {
        if (request.getContainerCase() != UpdateQuickLinksRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> {
                    Timestamp now = Timestamps.fromMillis(Instant.now().toEpochMilli());

                    QuickLinksValidator.validate(request);

                    String projectId = request.getProjectId();

                    var proto = QuickLinks.newBuilder()
                            .setProjectId(projectId)
                            .setData(request.getData())
                            .setModifiedAt(now)
                            .setModifiedBy(account.getId())
                            .build();

                    var model = QuickLinksDtoConverter.encode(proto, etag);

                    return checkForeignRefs(projectId)
                            .thenCompose(aVoid -> quickLinksDao.upsert(model))
                            .thenApply(upsertedOpt -> {
                                if (upsertedOpt.isEmpty()) {
                                    throw new ConflictException(String.format("quick links for project %s with version %s is out of date", projectId, model.getVersion()));
                                }
                                return QuickLinksDtoConverter.decode(upsertedOpt.get());
                            });
                });
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteQuickLinksRequest request, AuthSubject subject) {
        if (request.getContainerCase() != DeleteQuickLinksRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> quickLinksDao.delete(request.getProjectId()))
                .thenApply(deleted -> {
                    if (!deleted) {
                        throw quickLinksNotFound(request.getProjectId());
                    }
                    return Empty.getDefaultInstance();
                });
    }

    private CompletableFuture<Void> checkForeignRefs(String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new NotFoundException(String.format("project %s does not exist", projectId));
                    }
                });
    }

    private static NotFoundException quickLinksNotFound(String projectId) {
        return new NotFoundException(String.format("no quick links with id %s found", projectId));
    }
}
