package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateResourceRequest;
import ru.yandex.monitoring.api.v3.DeleteResourceRequest;
import ru.yandex.monitoring.api.v3.GetResourceRequest;
import ru.yandex.monitoring.api.v3.ListResourcesRequest;
import ru.yandex.monitoring.api.v3.ListResourcesResponse;
import ru.yandex.monitoring.api.v3.ServiceProviderResource;
import ru.yandex.monitoring.api.v3.UpdateResourceRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.v3.intranet.ResourceService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ServiceProviderResourceDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.validators.ServiceProviderResourceValidator;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.UpdateRequest;

/**
 * @author Alexey Trushkin
 */
//todo cloud and folder support
@Component
@ParametersAreNonnullByDefault
public class ResourceServiceImpl implements ResourceService {
    private final Authorizer authorizer;
    private final NameResolverClient nameResolverClient;

    @Autowired
    public ResourceServiceImpl(
            Authorizer authorizer,
            NameResolverClient nameResolverClient)
    {
        this.authorizer = authorizer;
        this.nameResolverClient = nameResolverClient;
    }

    @Override
    public CompletableFuture<ServiceProviderResource> get(GetResourceRequest request, AuthSubject subject) {
        ServiceProviderResourceValidator.validate(request);
        var req = ServiceProviderResourceDtoConverter.toResolveRequest(request);
        return nameResolverClient.resolve(req)
                .thenApply(resolveResponse -> ServiceProviderResourceDtoConverter.toSingleResource(resolveResponse.resources));
    }

    @Override
    public CompletableFuture<ServiceProviderResource> create(CreateResourceRequest request, AuthSubject subject) {
        ServiceProviderResourceValidator.validate(request);
        UpdateRequest req = ServiceProviderResourceDtoConverter.toUpdateResourcesRequest(request);
        return authorizer.authorize(subject, authorizationObject(request.getAbcSlug()), Permission.CONFIGS_CREATE)
                .thenCompose(account -> nameResolverClient.update(req))
                .thenApply(response -> ServiceProviderResourceDtoConverter.toServiceProviderResource(request));
    }

    @Override
    public CompletableFuture<ServiceProviderResource> update(UpdateResourceRequest request, AuthSubject subject) {
        ServiceProviderResourceValidator.validate(request);
        var getRequest = ServiceProviderResourceDtoConverter.toResolveRequest(GetResourceRequest.newBuilder()
                .setAbcSlug(request.getAbcSlug())
                .setId(request.getId())
                .build());

        return authorizer.authorize(subject, authorizationObject(request.getAbcSlug()), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> nameResolverClient.resolve(getRequest))
                .thenApply(resolveResponse -> ServiceProviderResourceDtoConverter.toUpdateResourcesRequest(request, resolveResponse))
                .thenCompose(req -> nameResolverClient.update(req)
                        .thenApply(response -> ServiceProviderResourceDtoConverter.toSingleResource(req.resources)));
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteResourceRequest request, AuthSubject subject) {
        ServiceProviderResourceValidator.validate(request);
        var getRequest = ServiceProviderResourceDtoConverter.toResolveRequest(GetResourceRequest.newBuilder()
                .setAbcSlug(request.getAbcSlug())
                .setId(request.getId())
                .build());
        return authorizer.authorize(subject, authorizationObject(request.getAbcSlug()), Permission.CONFIGS_DELETE)
                .thenCompose(account -> nameResolverClient.resolve(getRequest))
                .thenApply(resolveResponse -> ServiceProviderResourceDtoConverter.toUpdateResourcesRequest(request, resolveResponse))
                .thenCompose(req -> nameResolverClient.update(req)
                        .thenApply(ignore -> Empty.getDefaultInstance()));
    }

    @Override
    public CompletableFuture<Void> upsert(List<CreateResourceRequest> requests, boolean removeOther, String serviceProviderId) {
        if (requests.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        ResolveRequest request = ServiceProviderResourceDtoConverter.toResolveRequest(requests);
        return nameResolverClient.resolve(request)
                .thenCompose(resolveResponse -> {
                    return nameResolverClient.update(ServiceProviderResourceDtoConverter.toUpdateResourcesRequest(resolveResponse, requests, removeOther, serviceProviderId));
                });
    }

    @Override
    public CompletableFuture<ListResourcesResponse> list(ListResourcesRequest request, AuthSubject subject) {
        ServiceProviderResourceValidator.validate(request);
        FindRequest req = ServiceProviderResourceDtoConverter.toUpdateResourcesRequest(request);
        return nameResolverClient.find(req)
                .thenApply(ServiceProviderResourceDtoConverter::toResources);
    }

    private AuthorizationObject authorizationObject(String abcSlug) {
        return AuthorizationObject.abc(abcSlug, "administration");
    }
}
