package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializerFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.frontend.ResourceYtDataSourceConfig;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.gateway.api.v3.intranet.ResourceService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ServiceProviderResourceDtoConverter;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.yt.ytclient.proxy.TableReader;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.ReadTable;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ResourcesYtDataSource implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ResourcesYtDataSource.class);

    private final DistributedLock lock;
    private final Clock clock;
    private final long initialDelayMillis;
    private final long minReindexPeriod;
    private final MetricRegistry registry;
    private final ResourceService resourceService;
    private final YtClient ytClient;
    private final ResourceYtDataSourceConfig config;
    private volatile long latestReindexTs;
    private volatile boolean closed;
    private final ScheduledExecutorService timer;
    private final PingActorRunner actor;
    private final AsyncMetrics metrics;
    private final Map<String, AsyncMetrics> serviceProviderMetrics;
    private final ServiceProvidersDao serviceProvidersDao;

    public ResourcesYtDataSource(
            ServiceProvidersDao serviceProvidersDao,
            DistributedLock lock,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry,
            ResourceYtDataSourceConfig config,
            SecretProvider secretProvider,
            ResourceService resourceService)
    {
        this.registry = registry;
        this.resourceService = resourceService;
        var intervalMillis = TimeUnitConverter.millis(config.getRefreshInterval());
        this.serviceProvidersDao = serviceProvidersDao;
        this.lock = lock;
        this.clock = clock;
        this.initialDelayMillis = Duration.ofMinutes(1).toMillis();
        minReindexPeriod = intervalMillis / 2;
        this.timer = timer;
        this.metrics = new AsyncMetrics(registry, "resources.data_source", Labels.of("source", "yt"));
        serviceProviderMetrics = new HashMap<>();
        this.ytClient = YtClient.builder()
                .setCluster(config.getYtCluster())
                .setRpcCredentials(new RpcCredentials("robot-solomon", secretProvider.getSecret(config.getOAuthToken()).orElseThrow()))
                .build();
        this.config = config;
        this.actor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .operation("Fetch service provider resources")
                .pingInterval(Duration.ofMillis(intervalMillis))
                .backoffDelay(Duration.ofMinutes(1))
                .onPing(this::act)
                .build();
        acquireLock();
    }

    public CompletableFuture<Void> fetchResources() {
        return serviceProvidersDao.findAll()
                .thenCompose(serviceProviders -> {
                    List<CompletableFuture<YtLoaderResult>> ytFutures = new ArrayList<>();
                    for (var serviceProvider : serviceProviders) {
                        YtTableLoader loader = new YtTableLoader(serviceProvider.getId(), serviceProvider.isHasGlobalId());
                        ytFutures.add(loader.load()
                                .thenApply(tableRows -> new YtLoaderResult(serviceProvider.getId(), tableRows)));
                    }
                    return CompletableFutures.allOf(ytFutures)
                            .thenCompose(this::writeResults);
                });
    }

    private CompletableFuture<Void> writeResults(ListF<YtLoaderResult> rowLists) {
        CompletableFuture<Void> result = CompletableFuture.completedFuture(null);
        for (YtLoaderResult res : rowLists) {
            var rowList = res.rows;
            Map<String, List<TableRow>> collect = rowList.stream()
                    .collect(Collectors.groupingBy(tableRow -> tableRow.abc_slug));
            for (Map.Entry<String, List<TableRow>> stringListEntry : collect.entrySet()) {
                var cmd = ServiceProviderResourceDtoConverter.toCreateResourceRequests(stringListEntry);
                result = result.thenCompose(unused -> resourceService.upsert(cmd, true, res.serviceProviderId()));
            }
        }
        return result;
    }

    public CompletableFuture<Void> act(int attempt) {
        if (closed) {
            return completedFuture(null);
        }

        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }
        long reindexTs = clock.millis();
        if (minReindexPeriod > reindexTs - latestReindexTs) {
            return completedFuture(null);
        }

        var future = fetchResources()
                .thenAccept(ignore -> latestReindexTs = reindexTs);
        metrics.forFuture(future);
        return future;
    }

    public void schedule() {
        long delay = DurationUtils.randomize(initialDelayMillis);
        timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire ResourcesYtDataSourceMaster lock, seqNo {}", seqNo);
                schedule();
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose ResourcesYtDataSourceMaster lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    private AsyncMetrics getYtLoaderMetrics(String id) {
        return serviceProviderMetrics.computeIfAbsent(id,
                s -> new AsyncMetrics(registry, "resources.data_source", Labels.of("source", "yt", "service_provider", id)));
    }

    @Override
    public void close() {
        closed = true;
        actor.close();
        ytClient.close();
    }

    @YTreeObject
    public static class TableRow {
        public String abc_slug;
        public Map<String, String> resource_id;
        public String service_provider_id;
        public String resource_type;
        public String global_id;
        public String responsible;
        public String environment;
    }

    private record YtLoaderResult(String serviceProviderId, List<TableRow> rows) {}

    private class YtTableLoader {
        private final String serviceProvider;
        private final List<TableRow> result = new ArrayList<>();
        private final CompletableFuture<List<TableRow>> done = new CompletableFuture<>();
        private final YPath table;
        private final boolean hasGlobalId;

        public YtTableLoader(String serviceProvider, boolean hasGlobalId) {
            this.serviceProvider = serviceProvider;
            table = YPath.simple(config.getYtResourceTablePrefix() + serviceProvider);
            this.hasGlobalId = hasGlobalId;
        }

        public CompletableFuture<List<TableRow>> load() {
            getYtLoaderMetrics(serviceProvider).forFuture(done);
            ytClient.readTable(new ReadTable<>(table, YTreeObjectSerializerFactory.forClass(TableRow.class)))
                    .thenAccept(this::readNext)
                    .exceptionally(ex -> {
                        handleDone(ex);
                        return null;
                    });
            return done;
        }

        private void readNext(TableReader<TableRow> reader) {
            if (!reader.canRead()) {
                reader.close().whenComplete((unused, ex) -> handleDone(ex));
            } else {
                try {
                    List<TableRow> temp;
                    while ((temp = reader.read()) != null) {
                        result.addAll(temp);
                    }
                    reader.readyEvent()
                            .thenAcceptAsync((unused) -> readNext(reader))
                            .exceptionally(ex -> {
                                handleDone(ex);
                                return null;
                            });
                } catch (Exception ex) {
                    handleDone(ex);
                }
            }
        }

        private void handleDone(@Nullable Throwable ex) {
            if (ex == null) {
                if (!hasGlobalId) {
                    result.forEach(tableRow -> tableRow.global_id = "");
                }
                done.complete(result);
            } else {
                logger.error("Error while reading '" + serviceProvider + "' resources: ", ex);
                var cause = Throwables.getRootCause(ex);
                if (!cause.getMessage().contains("Error resolving path " + config.getYtResourceTablePrefix() + serviceProvider)) {
                    registry.rate("resources.data_source", Labels.of("source", "yt", "service_provider", serviceProvider)).inc();
                }
                done.complete(List.of());
            }
        }
    }
}
