package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateServiceRequest;
import ru.yandex.monitoring.api.v3.DeleteServiceRequest;
import ru.yandex.monitoring.api.v3.GetServiceRequest;
import ru.yandex.monitoring.api.v3.ListServiceClustersRequest;
import ru.yandex.monitoring.api.v3.ListServiceClustersResponse;
import ru.yandex.monitoring.api.v3.ListServicesRequest;
import ru.yandex.monitoring.api.v3.ListServicesResponse;
import ru.yandex.monitoring.api.v3.Service;
import ru.yandex.monitoring.api.v3.UpdateServiceRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ServicesManager;
import ru.yandex.solomon.gateway.api.v3.intranet.ServiceService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ServiceDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ShardAssociationDtoConverter;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ServiceServiceImpl implements ServiceService {
    private final Authorizer authorizer;
    private final ServicesManager servicesManager;

    @Autowired
    public ServiceServiceImpl(Authorizer authorizer, ServicesManager servicesManager) {
        this.authorizer = authorizer;
        this.servicesManager = servicesManager;
    }

    @Override
    public CompletableFuture<Pair<Service, Integer>> get(GetServiceRequest request, AuthSubject subject) {
        if (request.getContainerCase() != GetServiceRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<Service, Integer>> doGet(GetServiceRequest request) {
        return servicesManager.getService(request.getProjectId(), "", request.getServiceId())
                .thenApply(ServiceDtoConverter::fromModelWithVersion);
    }

    @Override
    public CompletableFuture<ListServicesResponse> list(ListServicesRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListServicesRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doList(request));
    }

    private CompletableFuture<ListServicesResponse> doList(ListServicesRequest request) {
        return servicesManager.getPagedServices(
                request.getProjectId(),
                "",
                (int) request.getPageSize(),
                request.getPageToken(),
                request.getFilter())
                .thenApply(response -> {
                    var pagedResult = response.map(ServiceDtoConverter::fromModel);
                    return ListServicesResponse.newBuilder()
                            .addAllServices(pagedResult.getItems())
                            .setNextPageToken(pagedResult.getNextPageToken())
                            .build();
                });
    }

    @Override
    public CompletableFuture<Service> create(CreateServiceRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(account -> doCreate(request, subject));
    }

    private CompletableFuture<Service> doCreate(CreateServiceRequest request, AuthSubject subject) {
        Instant now = Instant.now();
        var service = ServiceDtoConverter.toModel(request, subject.getUniqueId(), now);
        return servicesManager.createService(service)
                .thenApply(ServiceDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Service> update(UpdateServiceRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> doUpdate(request, subject.getUniqueId(), etag));
    }

    private CompletableFuture<Service> doUpdate(UpdateServiceRequest request, String login, int etag) {
        Instant now = Instant.now();
        var service = ServiceDtoConverter.toModel(request, login, now, etag);
        return servicesManager.updateService(service)
                .thenApply(ServiceDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteServiceRequest request, AuthSubject subject) {
        if (request.getContainerCase() != DeleteServiceRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> doDelete(request));
    }

    private CompletableFuture<Empty> doDelete(DeleteServiceRequest request) {
        return servicesManager.deleteService(request.getProjectId(), "", request.getServiceId())
                .thenApply(aVoid -> Empty.getDefaultInstance());
    }

    @Override
    public CompletableFuture<ListServiceClustersResponse> listClusters(
            ListServiceClustersRequest request,
            AuthSubject subject)
    {
        if (request.getContainerCase() != ListServiceClustersRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doListClusters(request));
    }

    private CompletableFuture<ListServiceClustersResponse> doListClusters(ListServiceClustersRequest request) {
        String projectId = request.getProjectId();

        var future = servicesManager.getServiceAssociations(
                projectId,
                "",
                request.getServiceId(),
                ShardAssociationDtoConverter.toModel(request.getFilterByModesList()),
                request.getFilterByName(),
                (int) request.getPageSize(),
                request.getPageToken());

        return future.thenApply(response -> ShardAssociationDtoConverter.fromServiceClustersModel(projectId, response));
    }
}
