package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateShardRequest;
import ru.yandex.monitoring.api.v3.DeleteShardRequest;
import ru.yandex.monitoring.api.v3.GetShardRequest;
import ru.yandex.monitoring.api.v3.ListShardTargetsRequest;
import ru.yandex.monitoring.api.v3.ListShardTargetsResponse;
import ru.yandex.monitoring.api.v3.ListShardsRequest;
import ru.yandex.monitoring.api.v3.ListShardsResponse;
import ru.yandex.monitoring.api.v3.Shard;
import ru.yandex.monitoring.api.v3.UpdateShardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ShardsManager;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.api.v3.intranet.ShardService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ShardDtoConverter;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.net.KnownDc;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ShardServiceImpl implements ShardService {
    private final Authorizer authorizer;
    private final ShardsManager shardsManager;
    private final ShardDtoConverter shardDtoConverter;

    @Autowired
    public ShardServiceImpl(Authorizer authorizer, ShardsManager shardsManager, CoremonClient coremonClient) {
        this.authorizer = authorizer;
        this.shardsManager = shardsManager;
        this.shardDtoConverter = new ShardDtoConverter(coremonClient);
    }

    @Override
    public CompletableFuture<Pair<Shard, Integer>> get(GetShardRequest request, AuthSubject subject) {
        if (request.getContainerCase() != GetShardRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<Shard, Integer>> doGet(GetShardRequest request) {
        return shardsManager.getShard(request.getProjectId(), "", request.getShardId(), false)
                .thenApply(ShardsManager.ShardExtended::shard)
                .thenApply(shardDtoConverter::fromModelWithVersion);
    }

    @Override
    public CompletableFuture<ListShardsResponse> list(ListShardsRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListShardsRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doList(request));
    }

    private CompletableFuture<ListShardsResponse> doList(ListShardsRequest request) {
        return shardsManager.getProjectShardsV3(
                request.getProjectId(),
                "",
                (int) request.getPageSize(),
                request.getPageToken(),
                request.getFilter())
                .thenApply(response -> {
                    var pagedResult = response.map(shardDtoConverter::fromModel);
                    return ListShardsResponse.newBuilder()
                            .addAllShards(pagedResult.getItems())
                            .setNextPageToken(pagedResult.getNextPageToken())
                            .build();
                });
    }

    @Override
    public CompletableFuture<Shard> create(CreateShardRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(account -> {
                    boolean canUpdateInternals = account.can(Permission.CONFIGS_UPDATE_INTERNALS);
                    return doCreate(request, canUpdateInternals, subject);
                });
    }

    private CompletableFuture<Shard> doCreate(CreateShardRequest request, boolean canUpdateInternals, AuthSubject subject) {
        Instant now = Instant.now();
        var shard = shardDtoConverter.toModel(request, subject.getUniqueId(), now);
        return shardsManager.createShard(shard, canUpdateInternals, false)
                .thenApply(ShardsManager.ShardExtended::shard)
                .thenApply(shardDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Shard> update(UpdateShardRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> {
                    boolean canUpdateInternals = account.can(Permission.CONFIGS_UPDATE_INTERNALS);
                    return doUpdate(request, canUpdateInternals, subject.getUniqueId(), etag);
                });
    }

    private CompletableFuture<Shard> doUpdate(UpdateShardRequest request, boolean canUpdateInternals, String login, int etag) {
        Instant now = Instant.now();
        var shard = shardDtoConverter.toModel(request, login, now, etag);
        return shardsManager.updateShard(shard, canUpdateInternals, false)
                .thenApply(ShardsManager.ShardExtended::shard)
                .thenApply(shardDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteShardRequest request, AuthSubject subject) {
        if (request.getContainerCase() != DeleteShardRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> doDelete(request));
    }

    private CompletableFuture<Empty> doDelete(DeleteShardRequest request) {
        return shardsManager.deleteShard(request.getProjectId(), "", request.getShardId())
                .thenApply(aVoid -> Empty.getDefaultInstance());
    }

    @Override
    public CompletableFuture<ListShardTargetsResponse> listTargets(
            ListShardTargetsRequest request,
            AuthSubject subject)
    {
        if (request.getContainerCase() != ListShardTargetsRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doListTargets(request));
    }

    private CompletableFuture<ListShardTargetsResponse> doListTargets(ListShardTargetsRequest request) {
        String filterByDc = request.getFilterByDc();
        String filterByStatus = request.getFilterByStatus();

        final KnownDc dc = shardDtoConverter.parseFilterByDc(filterByDc);

        boolean notOkStatus = "NOT_OK".equals(filterByStatus);
        final UrlStatusType status = shardDtoConverter.parseFilterByStatus(filterByStatus, notOkStatus);

        TokenPageOptions pageOptions = new TokenPageOptions((int) request.getPageSize(), request.getPageToken());

        var future = shardsManager.fetcherTargetsStatusV3(
                request.getProjectId(),
                "",
                request.getShardId(),
                request.getFetcherHost(),
                request.getFilterByHost(),
                dc,
                status,
                notOkStatus,
                pageOptions);

        return future.thenApply(shardDtoConverter::fromModel);
    }
}
