package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.ExplainSubAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainSubAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertEvaluationStateResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertNotificationStateResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertRequest;
import ru.yandex.monitoring.api.v3.ListSubAlertsRequest;
import ru.yandex.monitoring.api.v3.ListSubAlertsResponse;
import ru.yandex.monitoring.api.v3.SubAlert;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.TEvaluationState;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.v3.intranet.SubAlertService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.AlertDtoConverter;

import static ru.yandex.solomon.gateway.api.cloud.v1.RequestStatusToAlertingException.throwIfNotOk;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class SubAlertServiceImpl implements SubAlertService {
    private final Authorizer authorizer;
    private final AlertingClient alertingClient;

    @Autowired
    public SubAlertServiceImpl(Authorizer authorizer, AlertingClient alertingClient) {
        this.authorizer = authorizer;
        this.alertingClient = alertingClient;
    }

    @Override
    public CompletableFuture<SubAlert> get(GetSubAlertRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(aVoid -> doGet(request));
    }

    private CompletableFuture<SubAlert> doGet(GetSubAlertRequest request) {
        TReadSubAlertRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readSubAlert(readRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<ListSubAlertsResponse> list(ListSubAlertsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doList(request));
    }

    private CompletableFuture<ListSubAlertsResponse> doList(ListSubAlertsRequest request) {
        TListSubAlertRequest listRequest = AlertDtoConverter.toProto(request);

        return alertingClient.listSubAlerts(listRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<GetSubAlertEvaluationStateResponse> getEvaluationState(GetSubAlertEvaluationStateRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), request.getAlertId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetEvaluationState(request));
    }

    private CompletableFuture<GetSubAlertEvaluationStateResponse> doGetEvaluationState(GetSubAlertEvaluationStateRequest request) {
        TReadEvaluationStateRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readEvaluationState(readRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    TEvaluationState state = response.getState();
                    return AlertDtoConverter.fromProto(state);
                });
    }

    @Override
    public CompletableFuture<GetSubAlertNotificationStateResponse> getNotificationState(GetSubAlertNotificationStateRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), request.getAlertId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetNotificationState(request));
    }

    private CompletableFuture<GetSubAlertNotificationStateResponse> doGetNotificationState(GetSubAlertNotificationStateRequest request) {
        TReadNotificationStateRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readNotificationState(readRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromSubAlertProto(response);
                });
    }

    @Override
    public CompletableFuture<ExplainSubAlertEvaluationResponse> explainEvaluation(ExplainSubAlertEvaluationRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doExplainEvaluation(request));
    }

    private CompletableFuture<ExplainSubAlertEvaluationResponse> doExplainEvaluation(ExplainSubAlertEvaluationRequest request) {
        long evaluationTimeMillis = Timestamps.toMillis(request.getTime());
        if (evaluationTimeMillis == 0) {
            evaluationTimeMillis = System.currentTimeMillis();
        }

        long deadlineMillis = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L);

        TExplainEvaluationRequest explainRequest = AlertDtoConverter.toProto(request, evaluationTimeMillis, deadlineMillis);

        return alertingClient.explainEvaluation(explainRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromSubAlertProto(response);
                });
    }
}
