package ru.yandex.solomon.gateway.api.v3.intranet.priv.dto;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.api.v3.priv.ProjectAcl;
import ru.yandex.monitoring.api.v3.priv.UpdateProjectAclRequest;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ProjectAclDtoConverter {
    public static ProjectAcl fromModel(Project project) {
        List<ProjectAcl.AclEntry> entries = fromAclEntriesModel(project.getAcl());

        return ProjectAcl.newBuilder()
                .setOwner(project.getOwner())
                .addAllEntries(entries)
                .setAuthorizedReadOnly(project.isOnlyAuthRead())
                .setAuthorizedPushOnly(project.isOnlyAuthPush())
                .build();
    }

    public static Project toModel(UpdateProjectAclRequest request, Project project, int etag) {
        Acl acl = ProjectAclDtoConverter.toModel(request);

        return project.toBuilder()
                .setOwner(request.getOwner())
                .setAcl(acl)
                .setOnlyAuthRead(request.getAuthorizedReadOnly())
                .setOnlyAuthPush(request.getAuthorizedPushOnly())
                .setVersion(etag)
                .build();
    }

    private static Acl toModel(UpdateProjectAclRequest request) {
        HashSet<String> canRead = new HashSet<>();
        HashSet<String> canUpdate = new HashSet<>();
        HashSet<String> canDelete = new HashSet<>();
        HashSet<String> canWrite = new HashSet<>();

        for (ProjectAcl.AclEntry entry : request.getEntriesList()) {
            if (entry.getCanRead()) {
                canRead.add(entry.getLogin());
            }
            if (entry.getCanUpdate()) {
                canUpdate.add(entry.getLogin());
            }
            if (entry.getCanDelete()) {
                canDelete.add(entry.getLogin());
            }
            if (entry.getCanWrite()) {
                canWrite.add(entry.getLogin());
            }
        }

        return Acl.of(canRead, canUpdate, canDelete, canWrite);
    }

    private static List<ProjectAcl.AclEntry> fromAclEntriesModel(Acl acl) {
        HashMap<String, ProjectAcl.AclEntry.Builder> aclEntryByUser = new HashMap<>();

        acl.getCanRead().forEach(user -> getOrCreate(aclEntryByUser, user).setCanRead(true));
        acl.getCanUpdate().forEach(user -> getOrCreate(aclEntryByUser, user).setCanUpdate(true));
        acl.getCanDelete().forEach(user -> getOrCreate(aclEntryByUser, user).setCanDelete(true));
        acl.getCanWrite().forEach(user -> getOrCreate(aclEntryByUser, user).setCanWrite(true));

        return aclEntryByUser.values().stream()
                .map(ProjectAcl.AclEntry.Builder::build)
                .collect(Collectors.toList());
    }

    private static ProjectAcl.AclEntry.Builder getOrCreate(
            HashMap<String, ProjectAcl.AclEntry.Builder> aclEntryByUser,
            String user)
    {
        return aclEntryByUser.computeIfAbsent(user, user1 -> ProjectAcl.AclEntry.newBuilder().setLogin(user1));
    }
}
