package ru.yandex.solomon.gateway.api.v3.intranet.priv.impl;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.priv.GetProjectAclRequest;
import ru.yandex.monitoring.api.v3.priv.ProjectAcl;
import ru.yandex.monitoring.api.v3.priv.UpdateProjectAclRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.ProjectsManager;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.gateway.api.v3.intranet.priv.ProjectAclService;
import ru.yandex.solomon.gateway.api.v3.intranet.priv.dto.ProjectAclDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.priv.validators.ProjectAclValidator;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ProjectAclServiceImpl implements ProjectAclService {
    private final Authorizer authorizer;
    private final ProjectsManager projectsManager;
    private final ProjectAclValidator projectAclValidator;

    @Autowired
    public ProjectAclServiceImpl(
            Authorizer authorizer,
            ProjectsManager projectsManager)
    {
        this.authorizer = authorizer;
        this.projectsManager = projectsManager;
        this.projectAclValidator = new ProjectAclValidator();
    }

    @Override
    public CompletableFuture<Pair<ProjectAcl, Integer>> get(GetProjectAclRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<ProjectAcl, Integer>> doGet(GetProjectAclRequest request) {
        return projectsManager.getProject(request.getProjectId())
                .thenApply(response -> {
                    ProjectAcl projectAcl = ProjectAclDtoConverter.fromModel(response);
                    int version = response.getVersion();
                    return Pair.of(projectAcl, version);
                });
    }

    @Override
    public CompletableFuture<ProjectAcl> update(UpdateProjectAclRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> {
                    boolean canUpdateAny = account.getRoles().hasPermission(Permission.CONFIGS_UPDATE_ANY);
                    String login = subject.getUniqueId();
                    return doUpdate(request, login, canUpdateAny, etag);
                });
    }

    private CompletableFuture<ProjectAcl> doUpdate(UpdateProjectAclRequest request, String login, boolean canUpdateAny, int etag) {
        projectAclValidator.validate(request);

        return projectsManager.getProject(request.getProjectId())
                .thenCompose(project -> {
                    if (etag != -1 && etag != project.getVersion()) {
                        String message = String.format(
                                "project %s with version %s is out of date",
                                project.getId(),
                                project.getVersion()
                        );
                        return CompletableFuture.failedFuture(new ConflictException(message));
                    }

                    Project newProject = ProjectAclDtoConverter.toModel(request, project, etag);

                    return projectsManager.updateProject(login, newProject, canUpdateAny, true)
                            .thenApply(ProjectAclDtoConverter::fromModel);
                });
    }
}
