package ru.yandex.solomon.gateway.api.v3.intranet.priv.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.priv.GetProjectAclRequest;
import ru.yandex.monitoring.api.v3.priv.UpdateProjectAclRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.priv.ProjectAclService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ProjectAclV3Controller {
    private final ProjectAclService service;

    @Autowired
    public ProjectAclV3Controller(ProjectAclService projectAclService) {
        this.service = projectAclService;
    }

    @RequestMapping(path = "/api/v3/priv/projects/{projectId}/acl", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetProjectAclRequest request = GetProjectAclRequest.newBuilder()
                .setProjectId(projectId)
                .build();

        return service.get(request, subject)
                .thenApply(response -> {
                    var project = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(project, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/priv/projects/{projectId}/acl", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest httpRequest)
    {
        int etag = EtagInterceptor.parseEtagHeader(httpRequest);
        UpdateProjectAclRequest.Builder builder = UpdateProjectAclRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        UpdateProjectAclRequest request = builder.build();
        return service.update(request, subject, etag)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
