package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CompleteStubRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequest;
import ru.yandex.monitoring.api.v3.GetStubRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertingStubRequestService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Alexey Trushkin
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class AlertingStubRequestController {

    private final AlertingStubRequestService service;

    public AlertingStubRequestController(AlertingStubRequestService service) {
        this.service = service;
    }

    @RequestMapping(path = "/api/v3/stubs/{stubRequestId}/alerts", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("stubRequestId") String stubRequestId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateStubRequest.Builder builder = CreateStubRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setId(stubRequestId);
        CreateStubRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/stubs/{stubRequestId}/alerts", method = RequestMethod.GET)
    CompletableFuture<String> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("stubRequestId") String stubRequestId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetStubRequest getRequest = GetStubRequest.newBuilder()
                .setId(stubRequestId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/stubs/{stubRequestId}/alerts/complete", method = RequestMethod.POST)
    CompletableFuture<String> complete(
            @RequireAuth AuthSubject subject,
            @PathVariable("stubRequestId") String stubRequestId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CompleteStubRequest.Builder builder = CompleteStubRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setId(stubRequestId);
        CompleteStubRequest request = builder.build();
        return service.complete(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
