package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.Alert;
import ru.yandex.monitoring.api.v3.AlertType;
import ru.yandex.monitoring.api.v3.CreateAlertRequest;
import ru.yandex.monitoring.api.v3.DeleteAlertRequest;
import ru.yandex.monitoring.api.v3.EvaluationStatus;
import ru.yandex.monitoring.api.v3.ExplainAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainNewAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertRequest;
import ru.yandex.monitoring.api.v3.GetAlertStatsRequest;
import ru.yandex.monitoring.api.v3.ListAlertsRequest;
import ru.yandex.monitoring.api.v3.ListFullAlertsRequest;
import ru.yandex.monitoring.api.v3.MuteAlertRequest;
import ru.yandex.monitoring.api.v3.NotificationStatus;
import ru.yandex.monitoring.api.v3.UnmuteAlertRequest;
import ru.yandex.monitoring.api.v3.UpdateAlertRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class AlertsV3Controller {
    private final AlertService service;

    @Autowired
    public AlertsV3Controller(AlertService service) {
        this.service = service;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
                    @RequireAuth AuthSubject subject,
                    @PathVariable("projectId") String projectId,
                    @PathVariable("alertId") String alertId,
                    @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertRequest getRequest = GetAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var alert = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(alert, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts:listFull", method = RequestMethod.GET)
    CompletableFuture<String> listFull(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListFullAlertsRequest listRequest = ListFullAlertsRequest.newBuilder()
                .setProjectId(projectId)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.listFull(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filterByName", defaultValue = "", required = false) String filterByName,
            @RequestParam(value = "filterByStatuses", defaultValue = "", required = false) List<Alert.Status> filterByStatuses,
            @RequestParam(value = "filterByTypes", defaultValue = "", required = false) List<AlertType> filterByTypes,
            @RequestParam(value = "filterByEvaluationStatuses", defaultValue = "", required = false) List<EvaluationStatus> filterByEvaluationStatuses,
            @RequestParam(value = "filterByNotificationStatuses", defaultValue = "", required = false) List<NotificationStatus> filterByNotificationStatuses,
            @RequestParam(value = "filterByChannelIds", defaultValue = "", required = false) List<String> filterByChannelIds,
            @RequestParam(value = "orderByName", defaultValue = "SORT_ORDER_UNSPECIFIED", required = false) ListAlertsRequest.SortOrder orderByName,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListAlertsRequest listRequest = ListAlertsRequest.newBuilder()
                .setProjectId(projectId)
                .setFilterByName(filterByName)
                .addAllFilterByStatuses(filterByStatuses)
                .addAllFilterByTypes(filterByTypes)
                .addAllFilterByEvaluationStatuses(filterByEvaluationStatuses)
                .addAllFilterByNotificationStatuses(filterByNotificationStatuses)
                .addAllFilterByChannelIds(filterByChannelIds)
                .setOrderByName(orderByName)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateAlertRequest.Builder builder = CreateAlertRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateAlertRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateAlertRequest.Builder builder = UpdateAlertRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateAlertRequest updateRequest = builder
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> delete(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId)
    {
        DeleteAlertRequest deleteRequest = DeleteAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:mute", method = RequestMethod.POST)
    CompletableFuture<String> mute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        MuteAlertRequest muteRequest = MuteAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.mute(muteRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:getEvaluationStats", method = RequestMethod.GET)
    CompletableFuture<String> getEvaluationStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertEvaluationStatsRequest getEvalStatsRequest = GetAlertEvaluationStatsRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.getEvaluationStats(getEvalStatsRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:getNotificationStats", method = RequestMethod.GET)
    CompletableFuture<String> getNotificationStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertNotificationStatsRequest getNotificationStatsRequest = GetAlertNotificationStatsRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.getNotificationStats(getNotificationStatsRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:getEvaluationState", method = RequestMethod.GET)
    CompletableFuture<String> getEvaluationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertEvaluationStateRequest getEvalStateRequest = GetAlertEvaluationStateRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.getEvaluationState(getEvalStateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:getNotificationState", method = RequestMethod.GET)
    CompletableFuture<String> getNotificationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertNotificationStateRequest getNotificationStateRequest = GetAlertNotificationStateRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.getNotificationState(getNotificationStateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:explainEvaluation", method = RequestMethod.GET)
    CompletableFuture<String> explainEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "time", defaultValue = "") String time,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        long timeMillis = time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli();

        ExplainAlertEvaluationRequest explainRequest = ExplainAlertEvaluationRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .setTime(Timestamps.fromMillis(timeMillis))
                .build();

        return service.explainEvaluation(explainRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts:explainEvaluation", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> explainNewEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        ExplainNewAlertEvaluationRequest.Builder builder = ExplainNewAlertEvaluationRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        ExplainNewAlertEvaluationRequest request = builder.build();

        return service.explainNewEvaluation(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts:getStats", method = RequestMethod.GET)
    CompletableFuture<String> getStats(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetAlertStatsRequest getStatsRequest = GetAlertStatsRequest.newBuilder()
                .setProjectId(projectId)
                .build();

        return service.getStats(getStatsRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}:unmute", method = RequestMethod.POST)
    CompletableFuture<String> unmute(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UnmuteAlertRequest unmuteRequest = UnmuteAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return service.unmute(unmuteRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
