package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateChannelRequest;
import ru.yandex.monitoring.api.v3.DeleteChannelRequest;
import ru.yandex.monitoring.api.v3.GetChannelRequest;
import ru.yandex.monitoring.api.v3.ListChannelsRequest;
import ru.yandex.monitoring.api.v3.UpdateChannelRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.ChannelService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ChannelsV3Controller {
    private final ChannelService service;

    @Autowired
    public ChannelsV3Controller(ChannelService service) {
        this.service = service;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/channels/{channelId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("channelId") String channelId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetChannelRequest getRequest = GetChannelRequest.newBuilder()
                .setProjectId(projectId)
                .setChannelId(channelId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var channel = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(channel, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/channels", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "filterByType", defaultValue = "", required = false) List<ListChannelsRequest.ChannelType> filterByType,
            @RequestParam(value = "orderByName", defaultValue = "", required = false) ListChannelsRequest.SortOrder orderByName,
            @RequestParam(value = "orderByType", defaultValue = "", required = false) ListChannelsRequest.SortOrder orderByType,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListChannelsRequest listRequest = ListChannelsRequest.newBuilder()
                .setProjectId(projectId)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .addAllFilterByType(filterByType)
                .setOrderByName(orderByName)
                .setOrderByType(orderByType)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/channels", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateChannelRequest.Builder builder = CreateChannelRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateChannelRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/channels/{channelId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("channelId") String channelId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateChannelRequest.Builder builder = UpdateChannelRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateChannelRequest updateRequest = builder
                .setProjectId(projectId)
                .setChannelId(channelId)
                .build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/channels/{channelId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> delete(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("channelId") String channelId)
    {
        DeleteChannelRequest deleteRequest = DeleteChannelRequest.newBuilder()
                .setProjectId(projectId)
                .setChannelId(channelId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }
}
