package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateClusterRequest;
import ru.yandex.monitoring.api.v3.DeleteClusterRequest;
import ru.yandex.monitoring.api.v3.GetClusterRequest;
import ru.yandex.monitoring.api.v3.ListClusterServicesRequest;
import ru.yandex.monitoring.api.v3.ListClustersRequest;
import ru.yandex.monitoring.api.v3.Shard;
import ru.yandex.monitoring.api.v3.UpdateClusterRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.ClusterService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ClustersV3Controller {
    private final ClusterService service;

    @Autowired
    public ClustersV3Controller(ClusterService clusterService) {
        this.service = clusterService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListClustersRequest listRequest = ListClustersRequest.newBuilder()
                .setProjectId(projectId)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateClusterRequest.Builder builder = CreateClusterRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateClusterRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters/{clusterId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("clusterId") String clusterId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetClusterRequest getRequest = GetClusterRequest.newBuilder()
                .setProjectId(projectId)
                .setClusterId(clusterId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var cluster = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(cluster, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters/{clusterId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("clusterId") String clusterId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateClusterRequest.Builder builder = UpdateClusterRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateClusterRequest updateRequest = builder
                .setProjectId(projectId)
                .setClusterId(clusterId)
                .build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters/{clusterId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("clusterId") String clusterId)
    {
        DeleteClusterRequest deleteRequest = DeleteClusterRequest.newBuilder()
                .setProjectId(projectId)
                .setClusterId(clusterId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/clusters/{clusterId}/services", method = RequestMethod.GET)
    CompletableFuture<String> listServices(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("clusterId") String clusterId,
            @RequestParam(value = "filterByName", defaultValue = "") String filterByName,
            @RequestParam(value = "filterByModes", defaultValue = "") List<Shard.Mode> filterByModes,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListClusterServicesRequest listRequest = ListClusterServicesRequest.newBuilder()
                .setProjectId(projectId)
                .setClusterId(clusterId)
                .setFilterByName(filterByName)
                .addAllFilterByModes(filterByModes)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return service.listServices(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
