package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.ReadMetricsDataRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsDataService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class MetricsDataV3Controller {
    private final MetricsDataService metricsDataService;

    @Autowired
    public MetricsDataV3Controller(MetricsDataService metricsDataService) {
        this.metricsDataService = metricsDataService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/data/read", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    @ResponseBody
    CompletableFuture<String> read(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body) {
        ReadMetricsDataRequest.Builder builder = ReadMetricsDataRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        ReadMetricsDataRequest readRequest = builder.setProjectId(projectId).build();
        return metricsDataService.read(readRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/data/write", method = RequestMethod.POST)
    CompletableFuture<String> write(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam("cluster") String cluster,
            @RequestParam("service") String service,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest httpRequest) {
        String contentType = httpRequest.getHeaders().getFirst(HttpHeaders.CONTENT_TYPE);
        return metricsDataService.writeRawData(projectId, cluster, service, contentType, body, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
