package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.ListAllLabelValuesRequest;
import ru.yandex.monitoring.api.v3.ListLabelKeysRequest;
import ru.yandex.monitoring.api.v3.ListLabelValuesRequest;
import ru.yandex.monitoring.api.v3.ListMetricNamesRequest;
import ru.yandex.monitoring.api.v3.ListMetricsRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v2.managers.MetricsManager;
import ru.yandex.solomon.gateway.api.v3.intranet.MetricsMetaService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@Import({ MetricsManager.class})
@ParametersAreNonnullByDefault
public class MetricsMetaV3Controller {
    private final MetricsMetaService service;

    @Autowired
    public MetricsMetaV3Controller(MetricsMetaService service) {
        this.service = service;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/metrics", method = RequestMethod.GET)
    CompletableFuture<String> listMetrics(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "selectors", defaultValue = "") String selectors,
            @RequestParam(value = "useOldFormat", defaultValue = "false") boolean useOldFormat,
            @RequestParam(value = "pageSize", defaultValue = "0") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "forceZoneId", defaultValue = "") String forceZoneId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListMetricsRequest request = ListMetricsRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(selectors)
                .setUseOldFormat(useOldFormat)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .setForceZoneId(forceZoneId)
                .build();

        return service.listMetrics(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/metrics/names", method = RequestMethod.GET)
    CompletableFuture<String> listMetricNames(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "selectors", defaultValue = "") String selectors,
            @RequestParam(value = "nameFilter", defaultValue = "") String nameFilter,
            @RequestParam(value = "pageSize", defaultValue = "0") int pageSize,
            @RequestParam(value = "forceZoneId", defaultValue = "") String forceZoneId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListMetricNamesRequest request = ListMetricNamesRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(selectors)
                .setNameFilter(nameFilter)
                .setPageSize(pageSize)
                .setForceZoneId(forceZoneId)
                .build();

        return service.listMetricNames(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/metrics/labels", method = RequestMethod.GET)
    CompletableFuture<String> listLabelKeys(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "selectors", defaultValue = "") String selectors,
            @RequestParam(value = "useOldFormat", defaultValue = "false") boolean useOldFormat,
            @RequestParam(value = "forceZoneId", defaultValue = "") String forceZoneId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListLabelKeysRequest request = ListLabelKeysRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(selectors)
                .setUseOldFormat(useOldFormat)
                .setForceZoneId(forceZoneId)
                .build();

        return service.listLabelKeys(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/metrics/labels/{labelKey}/values", method = RequestMethod.GET)
    CompletableFuture<String> listLabelValues(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("labelKey") String labelKey,
            @RequestParam(value = "selectors", defaultValue = "") String selectors,
            @RequestParam(value = "useOldFormat", defaultValue = "false") boolean useOldFormat,
            @RequestParam(value = "valueFilter", defaultValue = "") String valueFilter,
            @RequestParam(value = "pageSize", defaultValue = "0") int pageSize,
            @RequestParam(value = "forceZoneId", defaultValue = "") String forceZoneId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListLabelValuesRequest request = ListLabelValuesRequest.newBuilder()
                .setProjectId(projectId)
                .setLabelKey(labelKey)
                .setSelectors(selectors)
                .setUseOldFormat(useOldFormat)
                .setValueFilter(valueFilter)
                .setPageSize(pageSize)
                .setForceZoneId(forceZoneId)
                .build();

        return service.listLabelValues(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/metrics/labelValues", method = RequestMethod.GET)
    CompletableFuture<String> listAllLabelValues(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "selectors", defaultValue = "") String selectors,
            @RequestParam(value = "useOldFormat", defaultValue = "false") boolean useOldFormat,
            @RequestParam(value = "valueFilter", defaultValue = "") String valueFilter,
            @RequestParam(value = "pageSize", defaultValue = "0") int pageSize,
            @RequestParam(value = "forceZoneId", defaultValue = "") String forceZoneId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListAllLabelValuesRequest request = ListAllLabelValuesRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(selectors)
                .setUseOldFormat(useOldFormat)
                .setValueFilter(valueFilter)
                .setPageSize(pageSize)
                .setForceZoneId(forceZoneId)
                .build();

        return service.listAllLabelValues(request, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
