package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.GetProjectRequest;
import ru.yandex.monitoring.api.v3.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.UpdateProjectRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.ProjectService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ProjectsV3Controller {
    private final ProjectService service;

    @Autowired
    public ProjectsV3Controller(ProjectService projectService) {
        this.service = projectService;
    }

    @RequestMapping(path = "/api/v3/projects", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListProjectsRequest listRequest = ListProjectsRequest.newBuilder()
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateProjectRequest.Builder builder = CreateProjectRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setId(projectId);
        CreateProjectRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetProjectRequest getRequest = GetProjectRequest.newBuilder()
                .setProjectId(projectId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var project = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(project, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateProjectRequest.Builder builder = UpdateProjectRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        UpdateProjectRequest updateRequest = builder.build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId)
    {
        DeleteProjectRequest deleteRequest = DeleteProjectRequest.newBuilder()
                .setProjectId(projectId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }
}
