package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateQuickLinksRequest;
import ru.yandex.monitoring.api.v3.DeleteQuickLinksRequest;
import ru.yandex.monitoring.api.v3.GetQuickLinksRequest;
import ru.yandex.monitoring.api.v3.UpdateQuickLinksRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.QuickLinksService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class QuickLinksV3Controller {
    private final QuickLinksService service;

    @Autowired
    public QuickLinksV3Controller(QuickLinksService quickLinksService) {
        this.service = quickLinksService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/quickLinks", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateQuickLinksRequest.Builder builder = CreateQuickLinksRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateQuickLinksRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/quickLinks", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetQuickLinksRequest getRequest = GetQuickLinksRequest.newBuilder()
                .setProjectId(projectId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var quickLinks = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(quickLinks, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/quickLinks", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateQuickLinksRequest.Builder builder = UpdateQuickLinksRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        UpdateQuickLinksRequest updateRequest = builder.build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/quickLinks", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> delete(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        DeleteQuickLinksRequest deleteRequest = DeleteQuickLinksRequest.newBuilder()
                .setProjectId(projectId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }
}
