package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateServiceRequest;
import ru.yandex.monitoring.api.v3.DeleteServiceRequest;
import ru.yandex.monitoring.api.v3.GetServiceRequest;
import ru.yandex.monitoring.api.v3.ListServiceClustersRequest;
import ru.yandex.monitoring.api.v3.ListServicesRequest;
import ru.yandex.monitoring.api.v3.Shard;
import ru.yandex.monitoring.api.v3.UpdateServiceRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.ServiceService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ServicesV3Controller {
    private final ServiceService service;

    @Autowired
    public ServicesV3Controller(ServiceService serviceService) {
        this.service = serviceService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListServicesRequest listRequest = ListServicesRequest.newBuilder()
                .setProjectId(projectId)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateServiceRequest.Builder builder = CreateServiceRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateServiceRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services/{serviceId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("serviceId") String serviceId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetServiceRequest getRequest = GetServiceRequest.newBuilder()
                .setProjectId(projectId)
                .setServiceId(serviceId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var service = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(service, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services/{serviceId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("serviceId") String serviceId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateServiceRequest.Builder builder = UpdateServiceRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateServiceRequest updateRequest = builder
                .setProjectId(projectId)
                .setServiceId(serviceId)
                .build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services/{serviceId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> deleteService(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("serviceId") String serviceId)
    {
        DeleteServiceRequest deleteRequest = DeleteServiceRequest.newBuilder()
                .setProjectId(projectId)
                .setServiceId(serviceId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/services/{serviceId}/clusters", method = RequestMethod.GET)
    CompletableFuture<String> listClusters(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("serviceId") String serviceId,
            @RequestParam(value = "filterByName", defaultValue = "") String filterByName,
            @RequestParam(value = "filterByModes", defaultValue = "") List<Shard.Mode> filterByModes,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListServiceClustersRequest listRequest = ListServiceClustersRequest.newBuilder()
                .setProjectId(projectId)
                .setServiceId(serviceId)
                .setFilterByName(filterByName)
                .addAllFilterByModes(filterByModes)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return service.listClusters(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
