package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.CreateShardRequest;
import ru.yandex.monitoring.api.v3.DeleteShardRequest;
import ru.yandex.monitoring.api.v3.GetShardRequest;
import ru.yandex.monitoring.api.v3.ListShardTargetsRequest;
import ru.yandex.monitoring.api.v3.ListShardsRequest;
import ru.yandex.monitoring.api.v3.UpdateShardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.intranet.ShardService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ShardsV3Controller {
    private final ShardService service;

    @Autowired
    public ShardsV3Controller(ShardService shardService) {
        this.service = shardService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListShardsRequest listRequest = ListShardsRequest.newBuilder()
                .setProjectId(projectId)
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards", method = RequestMethod.POST, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        CreateShardRequest.Builder builder = CreateShardRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setProjectId(projectId);
        CreateShardRequest request = builder.build();

        return service.create(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards/{shardId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("shardId") String shardId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetShardRequest getRequest = GetShardRequest.newBuilder()
                .setProjectId(projectId)
                .setShardId(shardId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var shard = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(shard, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards/{shardId}", method = RequestMethod.PATCH, consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("shardId") String shardId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        UpdateShardRequest.Builder builder = UpdateShardRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        UpdateShardRequest updateRequest = builder
                .setProjectId(projectId)
                .setShardId(shardId)
                .build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards/{shardId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<Void> delete(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("shardId") String shardId)
    {
        DeleteShardRequest deleteRequest = DeleteShardRequest.newBuilder()
                .setProjectId(projectId)
                .setShardId(shardId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(empty -> null);
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/shards/{shardId}/targets", method = RequestMethod.GET)
    CompletableFuture<String> listTargets(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("shardId") String shardId,
            @RequestParam(value = "fetcherHost", defaultValue = "") String fetcherHost,
            @RequestParam(value = "filterByHost", defaultValue = "") String filterByHost,
            @RequestParam(value = "filterByDc", defaultValue = "") String filterByDc,
            @RequestParam(value = "filterByStatus", defaultValue = "") String filterByStatus,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListShardTargetsRequest request = ListShardTargetsRequest.newBuilder()
                .setProjectId(projectId)
                .setShardId(shardId)
                .setFetcherHost(fetcherHost)
                .setFilterByHost(filterByHost)
                .setFilterByDc(filterByDc)
                .setFilterByStatus(filterByStatus)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return service.listTargets(request, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
