package ru.yandex.solomon.gateway.api.v3.intranet.rest;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monitoring.api.v3.EvaluationStatus;
import ru.yandex.monitoring.api.v3.ExplainSubAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertRequest;
import ru.yandex.monitoring.api.v3.ListSubAlertsRequest;
import ru.yandex.monitoring.api.v3.NotificationStatus;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.v3.intranet.SubAlertService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class SubAlertsV3Controller {
    private final SubAlertService service;

    @Autowired
    public SubAlertsV3Controller(SubAlertService subAlertService) {
        this.service = subAlertService;
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}/subAlerts/{subAlertId}", method = RequestMethod.GET)
    CompletableFuture<String> get(
                    @RequireAuth AuthSubject subject,
                    @PathVariable("projectId") String projectId,
                    @PathVariable("alertId") String alertId,
                    @PathVariable("subAlertId") String subAlertId,
                    @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetSubAlertRequest getRequest = GetSubAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .setSubAlertId(subAlertId)
                .build();
        return service.get(getRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}/subAlerts", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @RequestParam(value = "filterByEvaluationStatuses", defaultValue = "") List<EvaluationStatus> filterByEvaluationStatuses,
            @RequestParam(value = "filterByNotificationStatuses", defaultValue = "") List<NotificationStatus> filterByNotificationStatuses,
            @RequestParam(value = "filterByChannelIds", defaultValue = "") List<String> filterByChannelIds,
            @RequestParam(value = "filterByLabels", defaultValue = "") String filterByLabels,
            @RequestParam(value = "annotationKeys", defaultValue = "") List<String> annotationKeys,
            @RequestParam(value = "orderByLabels", defaultValue = "SORT_ORDER_UNSPECIFIED", required = false) ListSubAlertsRequest.SortOrder orderByLabels,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        ListSubAlertsRequest listRequest = ListSubAlertsRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .addAllFilterByEvaluationStatuses(filterByEvaluationStatuses)
                .addAllFilterByNotificationStatuses(filterByNotificationStatuses)
                .addAllFilterByChannelIds(filterByChannelIds)
                .setFilterByLabels(filterByLabels)
                .addAllAnnotationKeys(annotationKeys)
                .setOrderByLabels(orderByLabels)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}/subAlerts/{subAlertId}:getEvaluationState", method = RequestMethod.GET)
    CompletableFuture<String> getEvaluationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @PathVariable("subAlertId") String subAlertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetSubAlertEvaluationStateRequest getEvalStateRequest = GetSubAlertEvaluationStateRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .setSubAlertId(subAlertId)
                .build();

        return service.getEvaluationState(getEvalStateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}/subAlerts/{subAlertId}:getNotificationState", method = RequestMethod.GET)
    CompletableFuture<String> getNotificationState(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @PathVariable("subAlertId") String subAlertId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        GetSubAlertNotificationStateRequest getNotificationStateRequest = GetSubAlertNotificationStateRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .setSubAlertId(subAlertId)
                .build();

        return service.getNotificationState(getNotificationStateRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/api/v3/projects/{projectId}/alerts/{alertId}/subAlerts/{subAlertId}:explainEvaluation", method = RequestMethod.GET)
    CompletableFuture<String> explainEvaluation(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @PathVariable("alertId") String alertId,
            @PathVariable("subAlertId") String subAlertId,
            @RequestParam(value = "time", defaultValue = "") String time,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        long timeMillis = time.isEmpty() ? System.currentTimeMillis() : Instant.parse(time).toEpochMilli();

        ExplainSubAlertEvaluationRequest explainRequest = ExplainSubAlertEvaluationRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .setSubAlertId(subAlertId)
                .setTime(Timestamps.fromMillis(timeMillis))
                .build();

        return service.explainEvaluation(explainRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
