package ru.yandex.solomon.gateway.api.v3.intranet.validators;

import java.util.List;
import java.util.Map;

import ru.yandex.monitoring.api.v3.AlertConfig;
import ru.yandex.monitoring.api.v3.AlertParameter;
import ru.yandex.monitoring.api.v3.CompleteStubRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequest;
import ru.yandex.monitoring.api.v3.GetStubRequest;
import ru.yandex.monitoring.api.v3.MonitoringConfig;
import ru.yandex.monitoring.api.v3.NotificationChannel;
import ru.yandex.monitoring.api.v3.Resource;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Nuradil Zhambyl
 */
public class AlertingStubRequestValidator {

    public static void validate(GetStubRequest request) {
        validateId(request.getId(), "stub request");
    }

    public static void validate(CompleteStubRequest request) {
        validateId(request.getId(), "stub request");
        validateResources(request.getResourcesList());
    }

    public static void validate(CreateStubRequest request, SolomonConfHolder confHolder) {
        validateId(request.getId(), "stub request");
        validateId(request.getServiceProviderId(), "service provider");
        validateServiceProviderId(request.getServiceProviderId(), confHolder);
        validateStub(request.getStub(), confHolder);
    }

    private static void validateServiceProviderId(String serviceProviderId, SolomonConfHolder confHolder) {
        if (confHolder.getConfOrThrow().getServiceProvider(serviceProviderId) == null) {
            throw new BadRequestException("service provider id " + serviceProviderId + " invalid");
        }
    }

    private static void validateStub(MonitoringConfig stub, SolomonConfHolder confHolder) {
        validateId(stub.getProjectId(), "project");
        validateProject(stub.getProjectId(), confHolder);
        validateAlertConfigs(stub.getAlertConfigsList());
    }

    private static void validateProject(String projectId, SolomonConfHolder confHolder) {
        if (!confHolder.getConfOrThrow().projects().contains(projectId)) {
            throw new BadRequestException("project id " + projectId + " not found");
        }
    }

    private static void validateAlertConfigs(List<AlertConfig> alertConfigs) {
        for (var alertConfig : alertConfigs) {
            validateId(alertConfig.getAlertTemplateId(), "alert template");
            validateNotificationChannel(alertConfig.getNotificationChannelsList());
            validateAlertThreshold(alertConfig.getAlertThresholdsList());
        }
    }

    private static void validateAlertThreshold(List<AlertParameter> alertThresholds) {
        for (var alertThreshold : alertThresholds) {
            switch (alertThreshold.getParameterCase()) {
                case DOUBLE_PARAMETER_VALUE -> validateName(alertThreshold.getDoubleParameterValue().getName(), "double parameter");
                case INTEGER_PARAMETER_VALUE -> validateName(alertThreshold.getIntegerParameterValue().getName(), "integer parameter");
                case TEXT_PARAMETER_VALUE -> validateName(alertThreshold.getTextParameterValue().getName(), "text parameter");
                case TEXT_LIST_PARAMETER_VALUE -> validateName(alertThreshold.getTextListParameterValue().getName(), "text list parameter");
                case LABEL_LIST_PARAMETER_VALUE -> validateName(alertThreshold.getLabelListParameterValue().getName(), "label list parameter");
            }
        }
    }

    private static void validateName(String name, String object) {
        if (name.isBlank()) {
            throw new BadRequestException(object + " name cannot be blank");
        }
    }

    private static void validateNotificationChannel(List<NotificationChannel> notificationChannels) {
        if (notificationChannels.isEmpty()) {
            throw new BadRequestException("notification channel cannot be empty");
        }
        for (var notificationChannel : notificationChannels) {
            validateId(notificationChannel.getNotificationChannelId(), "notification channel");
        }
    }

    private static boolean invalidMap(Map<String, String> map) {
        for (Map.Entry<String, String> entry : map.entrySet()) {
            var key = entry.getKey();
            var value = entry.getValue();
            if (key.isBlank() || value.isBlank()) {
                return true;
            }
        }
        return map.isEmpty();
    }

    private static void validateResources(List<Resource> resources) {
        if (resources.isEmpty()) {
            throw new BadRequestException("resources cannot be empty");
        }
        for (var resource : resources) {
            if (invalidMap(resource.getResourceParametersMap())) {
                throw new BadRequestException("invalid resources");
            }
        }
    }

    private static void validateId(String id, String object) {
        if (id.isBlank()) {
            throw new BadRequestException(object + " ID cannot be blank");
        }
    }
}
