package ru.yandex.solomon.gateway.api.v3.intranet.validators;

import java.util.HashSet;
import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.api.v3.ChartWidget;
import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.api.v3.GetDashboardRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsRequest;
import ru.yandex.monitoring.api.v3.Parameter;
import ru.yandex.monitoring.api.v3.Parametrization;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.monitoring.api.v3.Widget;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DashboardValidator {
    private static final Pattern NAME_PATTERN = Pattern.compile("[a-z]([-a-z0-9]{0,61}[a-z0-9])?");

    public static void validate(GetDashboardRequest request) {
        if (request.getDashboardId().isBlank()) {
            throw new BadRequestException("dashboard ID cannot be blank");
        }
    }

    public static void validate(ListDashboardsRequest request) {
        if (!request.hasProjectId() && !request.hasFolderId()) {
            throw new BadRequestException("container ID cannot be blank");
        }
        if (!request.getFilter().isEmpty()) {
            try {
                var selector = Selectors.parse(request.getFilter());
                if (selector.size() != 1) {
                    throw new BadRequestException("Dashboard filter supports only name filter");
                }
                if (selector.findByKey("name") == null) {
                    throw new BadRequestException("Dashboard filter supports only name filter");
                }
            } catch (Exception e) {
                throw new BadRequestException("Dashboard filter wrong format");
            }
        }
    }

    public static void validate(CreateDashboardRequest request) {
        if (!request.getName().isBlank()) {
            if (!NAME_PATTERN.matcher(request.getName()).matches()) {
                throw new BadRequestException("name should match " + NAME_PATTERN.pattern());
            }
        }

        if (!request.hasProjectId() && !request.hasFolderId()) {
            throw new BadRequestException("container ID cannot be blank");
        }

        validate(request.getWidgetsList());
        validate(request.getParametrization());
    }

    public static void validate(UpdateDashboardRequest request) {
        if (request.getDashboardId().isBlank()) {
            throw new BadRequestException("dashboard ID cannot be blank");
        }
        if (!request.getName().isBlank()) {
            if (!NAME_PATTERN.matcher(request.getName()).matches()) {
                throw new BadRequestException("name should match " + NAME_PATTERN.pattern());
            }
        }

        validate(request.getWidgetsList());
        validate(request.getParametrization());
    }

    public static void validate(DeleteDashboardRequest request) {
        if (request.getDashboardId().isBlank()) {
            throw new BadRequestException("dashboard ID cannot be blank");
        }
    }

    private static void validate(List<Widget> widgets) {
        for (int i = 0; i < widgets.size(); i++) {
            Widget widget = widgets.get(i);
            try {
                validate(widget);
            } catch (BadRequestException e) {
                throw new BadRequestException(String.format("widget %s: %s", i + 1, e.getMessage()));
            }
        }
        validateChartIdUniqueness(widgets);
        validatePositionIntersections(widgets);
    }

    private static void validate(Widget widget) {
        validate(widget.getPosition());

        switch (widget.getWidgetCase()) {
            case TITLE -> {
                var title = widget.getTitle();
                if (title.getText().isBlank()) {
                    throw new BadRequestException("title widget text cannot be blank");
                }
            }
            case CHART -> {
                var chart = widget.getChart();
                if (chart.hasQueries()) {
                    ChartWidget.Queries queries = chart.getQueries();
                    queries.getTargetsList().forEach(DashboardValidator::validate);
                }
                if (chart.hasNameHidingSettings()) {
                    ChartWidget.NameHidingSettings nameHidingSettings = chart.getNameHidingSettings();
                    nameHidingSettings.getNamesList().forEach(name -> {
                        if (name.isBlank()) {
                            throw new BadRequestException("name in name hiding settings cannot be blank");
                        }
                    });
                }
            }
            case ALERT -> {
                var alert = widget.getAlert();
                if (alert.getProjectId().isBlank() && alert.getFolderId().isBlank()) {
                    throw new BadRequestException("container ID in alert widget cannot be blank");
                }
                if (alert.getAlertId().isBlank()) {
                    throw new BadRequestException("alert ID in alert widget cannot be blank");
                }
                validateSelectors(alert.getLabelsFilter(), "alert widget");
                alert.getAnnotationKeysList().forEach(key -> {
                    if (key.isBlank()) {
                        throw new BadRequestException("annotation key value in alert widget cannot be blank");
                    }
                });
            }
            case IFRAME -> {
                var iframe = widget.getIframe();
                if (iframe.getUrl().isBlank()) {
                    throw new BadRequestException("iframe widget URL cannot be blank");
                }
            }
        }
    }

    private static void validate(Widget.LayoutPosition position) {
        if (position.getX() < 0) {
            throw new BadRequestException("x in layout position cannot be negative");
        }
        if (position.getY() < 0) {
            throw new BadRequestException("y in layout position cannot be negative");
        }
        if (position.getW() <= 0) {
            throw new BadRequestException("w in layout position cannot be not positive");
        }
        if (position.getH() <= 0) {
            throw new BadRequestException("h in layout position cannot be not positive");
        }
    }

    private static void validatePositionIntersections(List<Widget> widgets) {
        if (widgets.size() < 2) {
            return;
        }
        for (int i = 0; i < widgets.size() - 1; ++i) {
            Widget.LayoutPosition position1 = widgets.get(i).getPosition();
            for (int j = i + 1; j < widgets.size(); ++j) {
                Widget.LayoutPosition position2 = widgets.get(j).getPosition();
                if (checkIntersection(position1, position2)) {
                    throw new BadRequestException(String.format("widgets %s and %s intersect", i + 1, j + 1));
                }
            }
        }
    }

    static boolean checkIntersection(Widget.LayoutPosition p1, Widget.LayoutPosition p2) {
        long x11 = p1.getX();
        long x12 = p1.getX() + p1.getW();
        long y11 = p1.getY();
        long y12 = p1.getY() + p1.getH();
        long x21 = p2.getX();
        long x22 = p2.getX() + p2.getW();
        long y21 = p2.getY();
        long y22 = p2.getY() + p2.getH();

        return x11 < x22 && x12 > x21 && y11 < y22 && y12 > y21;
    }

    private static void validate(ChartWidget.Queries.Target target) {
        if (target.getQuery().isBlank()) {
            throw new BadRequestException("target query cannot be blank");
        }
    }

    private static void validate(Parametrization parametrization) {
        validateSelectors(parametrization.getSelectors(), "parametrization");

        List<Parameter> parametersList = parametrization.getParametersList();

        parametersList.forEach(DashboardValidator::validate);
        validateParameterUniqueness(parametersList);
    }

    private static void validateParameterUniqueness(List<Parameter> parametersList) {
        HashSet<String> knownParameterNames = new HashSet<>(parametersList.size());
        for (Parameter parameter : parametersList) {
            String name = parameter.getName();
            if (knownParameterNames.contains(name)) {
                throw new BadRequestException("parameter with name " + name + " already exists");
            }
            knownParameterNames.add(name);
        }
    }

    private static void validateChartIdUniqueness(List<Widget> widgets) {
        HashSet<String> knownChartIds = new HashSet<>(widgets.size());
        for (Widget widget : widgets) {
            if (widget.hasChart()) {
                String id = widget.getChart().getId();
                if (knownChartIds.contains(id)) {
                    throw new BadRequestException("chart widget ID " + id + " already exists");
                }
                knownChartIds.add(id);
            }
        }
    }

    private static void validate(Parameter parameter) {
        if (parameter.getName().isBlank()) {
            throw new BadRequestException("parameter name cannot be blank");
        }
        if (parameter.getDataCase() == Parameter.DataCase.LABEL_VALUES) {
            var labelValues = parameter.getLabelValues();
            if (labelValues.getLabelKey().isBlank()) {
                throw new BadRequestException("key in label values parameter cannot be blank");
            }
            if (!labelValues.getSelectors().isBlank()) {
                validateSelectors(labelValues.getSelectors(), "label values parameter");
            }
        }
    }

    private static void validateSelectors(String selectors, String field) {
        if (!selectors.isBlank()) {
            try {
                Selectors.parse(selectors);
            } catch (SelectorsException e) {
                throw new BadRequestException("invalid selectors format in " + field + ": " + selectors);
            }
        }
    }
}
