package ru.yandex.solomon.gateway.api.v3.utils;

import java.nio.charset.StandardCharsets;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.protobuf.util.JsonFormat;

import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ProtoJsonUtils {
    private static final JsonFormat.TypeRegistry typeRegistry = JsonFormat.TypeRegistry
            .newBuilder()
            // Public Cloud API
            .add(yandex.cloud.api.monitoring.v3.AlertServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.api.monitoring.v3.ChannelServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.api.monitoring.v3.DashboardServiceOuterClass.getDescriptor().getMessageTypes())
            // Private Cloud API
            .add(yandex.cloud.priv.monitoring.v3.AlertServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ChannelServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ClusterServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.DashboardServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.QuickLinksServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ServiceDashboardServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ServiceServiceOuterClass.getDescriptor().getMessageTypes())
            .add(yandex.cloud.priv.monitoring.v3.ShardServiceOuterClass.getDescriptor().getMessageTypes())
            .build();

    private static final JsonFormat.Printer prettyJsonPrinter = JsonFormat.printer()
            .usingTypeRegistry(typeRegistry)
            .includingDefaultValueFields();

    private static final JsonFormat.Printer jsonPrinter = prettyJsonPrinter.omittingInsignificantWhitespace();

    public static <T extends Message.Builder> void fromJson(byte[] body, T builder) {
        try {
            String json = new String(body, StandardCharsets.UTF_8);
            JsonFormat.parser().ignoringUnknownFields().merge(json, builder);
        } catch (InvalidProtocolBufferException e) {
            throw new BadRequestException("failed to parse request body: " + e.getMessage());
        }
    }

    public static String toJson(Message response, boolean pretty) {
        try {
            var printer = pretty ? prettyJsonPrinter : jsonPrinter;
            return printer.print(response);
        } catch (InvalidProtocolBufferException e) {
            throw new BadRequestException("failed to format response to JSON: " + e.getMessage());
        }
    }
}
