package ru.yandex.solomon.gateway.api.v3alpha;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.CreateFavoriteRequest;
import ru.yandex.monitoring.v3.DeleteFavoriteRequest;
import ru.yandex.monitoring.v3.Favorite;
import ru.yandex.monitoring.v3.FavoriteServiceGrpc;
import ru.yandex.monitoring.v3.ListFavoriteRequest;
import ru.yandex.monitoring.v3.ListFavoriteResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcMonitoringFavoriteService extends FavoriteServiceGrpc.FavoriteServiceImplBase implements GrpcService {
    private final MonitoringFavoriteService monitoringFavoriteService;

    @Autowired
    public GrpcMonitoringFavoriteService(MonitoringFavoriteService monitoringFavoriteService) {
        this.monitoringFavoriteService = monitoringFavoriteService;
    }

    @Override
    public void list(ListFavoriteRequest request, StreamObserver<ListFavoriteResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(this::listImpl)
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ListFavoriteResponse> listImpl() {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return monitoringFavoriteService.list(authSubject);
    }

    @Override
    public void create(CreateFavoriteRequest request, StreamObserver<Favorite> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> createImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Favorite> createImpl(CreateFavoriteRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return monitoringFavoriteService.create(request, authSubject);
    }

    @Override
    public void delete(DeleteFavoriteRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> deleteImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> deleteImpl(DeleteFavoriteRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return monitoringFavoriteService.delete(request, authSubject);
    }
}
