package ru.yandex.solomon.gateway.api.v3alpha;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.GetQuickLinksRequest;
import ru.yandex.monitoring.v3.QuickLinks;
import ru.yandex.monitoring.v3.QuickLinksServiceGrpc;
import ru.yandex.monitoring.v3.UpdateQuickLinksRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcMonitoringQuickLinksService extends QuickLinksServiceGrpc.QuickLinksServiceImplBase implements GrpcService {
    private final Authorizer authorizer;
    private final MonitoringQuickLinksService service;

    @Autowired
    public GrpcMonitoringQuickLinksService(Authorizer authorizer, MonitoringQuickLinksService service) {
        this.authorizer = authorizer;
        this.service = service;
    }

    @Override
    public void get(GetQuickLinksRequest request, StreamObserver<QuickLinks> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<QuickLinks> getImpl(GetQuickLinksRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        var authFuture =
            authorizer.authorize(authSubject, request.getProjectId(), Permission.CONFIGS_GET);
        return authFuture.thenCompose(account -> service.getQuickLinks(request));
    }

    @Override
    public void update(UpdateQuickLinksRequest request, StreamObserver<QuickLinks> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> updateImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<QuickLinks> updateImpl(UpdateQuickLinksRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        var authFuture =
            authorizer.authorize(authSubject, request.getProjectId(), Permission.CONFIGS_UPDATE);
        return authFuture.thenCompose(account -> service.updateQuickLinks(request, account));
    }
}
