package ru.yandex.solomon.gateway.api.v3alpha;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.v3.CreateDashboardRequest;
import ru.yandex.monitoring.v3.Dashboard;
import ru.yandex.monitoring.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.v3.GetDashboardRequest;
import ru.yandex.monitoring.v3.ListDashboardRequest;
import ru.yandex.monitoring.v3.ListDashboardResponse;
import ru.yandex.monitoring.v3.UpdateDashboardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.conf.db3.ConfigV3DaoContext;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.MonitoringFavoritesDao;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.conf.db3.ydb.FavoriteRecord;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.MonitoringDashboardConverter;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@Import(ConfigV3DaoContext.class)
@Component
@ParametersAreNonnullByDefault
public class MonitoringDashboardServiceImpl implements MonitoringDashboardService {
    private final Authorizer authorizer;
    private final MonitoringDashboardsDao dashboardsDao;
    private final MonitoringFavoritesDao monitoringFavoritesDao;

    @Autowired
    public MonitoringDashboardServiceImpl(
            Authorizer authorizer,
            MonitoringDashboardsDao dashboardsDao,
            MonitoringFavoritesDao monitoringFavoritesDao)
    {
        this.authorizer = authorizer;
        this.dashboardsDao = dashboardsDao;
        this.monitoringFavoritesDao = monitoringFavoritesDao;
    }

    @Override
    public CompletableFuture<Dashboard> getDashboard(GetDashboardRequest request, AuthSubject authSubject) {
        String dashboardId = request.getDashboardId();

        return withFavorite(authSubject.getUniqueId(), dashboardsDao.readById(dashboardId))
                .thenApply(entityOpt -> {
                    if (entityOpt.isEmpty()) {
                        throw dashboardNotFound(dashboardId);
                    }
                    return entityOpt.get();
                })
                .thenCompose(dashboard -> authorizer.authorize(authSubject, dashboard.getProjectId(), Permission.CONFIGS_GET)
                        .thenApply(account -> dashboard));
    }

    @Override
    public CompletableFuture<Optional<Dashboard>> getDashboard(GetDashboardRequest request) {
        String dashboardId = request.getDashboardId();
        return dashboardsDao.readById(dashboardId)
                .thenApply(entity -> {
                    if (entity.isEmpty()) {
                        return Optional.empty();
                    }
                    return Optional.of(MonitoringDashboardConverter.decode(entity.get()));
                });
    }

    @Override
    public CompletableFuture<ListDashboardResponse> listDashboards(ListDashboardRequest request, AuthSubject authSubject) {
        String projectId = request.getProjectId();
        if (projectId.isBlank()) {
            return CompletableFuture.failedFuture(new BadRequestException("projectId must be specified"));
        }

        return authorizer.authorize(authSubject, projectId, Permission.CONFIGS_LIST)
                .thenCompose(account -> withFavorites(authSubject.getUniqueId(), dashboardsDao.list(projectId, request.getFilter(), (int) request.getPageSize(), request.getPageToken()))
                        .thenApply(pagedResult -> ListDashboardResponse.newBuilder()
                                .setNextPageToken(pagedResult.getNextPageToken())
                                .addAllDashboards(pagedResult.getItems())
                                .build()));
    }

    @Override
    public CompletableFuture<Dashboard> createDashboard(CreateDashboardRequest request, AuthSubject authSubject) {
        Instant now = Instant.now();

        return authorizer.authorize(authSubject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> {
                    String projectId = request.getProjectId();

                    String dashboardId = IdGenerator.generateInternalId();

                    Dashboard dashboard = Dashboard.newBuilder()
                            .setProjectId(projectId)
                            .setId(dashboardId)
                            .setName(request.getName())
                            .setDescription(request.getDescription())
                            .addAllWidgets(request.getWidgetsList())
                            .setParametrization(request.getParametrization())
                            .setCreatedAt(Timestamps.fromMillis(now.toEpochMilli()))
                            .setUpdatedAt(Timestamps.fromMillis(now.toEpochMilli()))
                            .setCreatedBy(authSubject.getUniqueId())
                            .setUpdatedBy(authSubject.getUniqueId())
                            .putAllLabels(request.getLabelsMap())
                            .setTitle(request.getTitle())
                            .setVersion(0)
                            .build();

                    Entity entity = MonitoringDashboardConverter.encode(dashboard, "");

                    return dashboardsDao.insert(entity)
                            .thenApply(inserted -> {
                                if (!inserted) {
                                    throw new ConflictException("dashboard " + dashboardId + " already exists");
                                }
                                return dashboard;
                            });
                });
    }

    @Override
    public CompletableFuture<Dashboard> updateDashboard(UpdateDashboardRequest request, AuthSubject authSubject) {
        return dashboardsDao.readById(request.getDashboardId())
                .thenCompose(optionalEntity -> withFavorite(authSubject.getUniqueId(), CompletableFuture.completedFuture(optionalEntity))
                        .thenCompose(existingDashboardOpt -> {
                            if (existingDashboardOpt.isEmpty()) {
                                return CompletableFuture.failedFuture(dashboardNotFound(request.getDashboardId()));
                            }

                            var existingDashboard = existingDashboardOpt.get();

                            String projectId = existingDashboard.getProjectId();

                            return authorizer.authorize(authSubject, projectId, Permission.CONFIGS_UPDATE)
                                    .thenCompose(account -> {
                                        Timestamp now = Timestamps.fromMillis(Instant.now().toEpochMilli());

                                        Dashboard dashboard = Dashboard.newBuilder()
                                                .setProjectId(projectId)
                                                .setId(request.getDashboardId())
                                                .setName(request.getName())
                                                .setDescription(request.getDescription())
                                                .addAllWidgets(request.getWidgetsList())
                                                .setParametrization(request.getParametrization())
                                                .setUpdatedAt(now)
                                                .setUpdatedBy(authSubject.getUniqueId())
                                                .putAllLabels(request.getLabelsMap())
                                                .setTitle(request.getTitle())
                                                .setVersion(request.getVersion())
                                                .setCreatedAt(existingDashboard.getCreatedAt())
                                                .setCreatedBy(existingDashboard.getCreatedBy())
                                                .build();

                                        Entity entity = MonitoringDashboardConverter.encode(dashboard, optionalEntity.get().getLocalId());
                                        return dashboardsDao.update(entity)
                                                .thenApply(updatedEntityOpt -> updatedEntityOpt.map(MonitoringDashboardConverter::decode))
                                                .thenCompose(updatedDashboardOpt -> checkExistence(dashboard, updatedDashboardOpt));
                                    });
                        }));
    }

    private CompletableFuture<Dashboard> checkExistence(
        Dashboard dashboard,
        Optional<Dashboard> updatedDashboardOpt)
    {
        if (updatedDashboardOpt.isPresent()) {
            return CompletableFuture.completedFuture(updatedDashboardOpt.get());
        }

        String projectId = dashboard.getProjectId();
        String dashboardId = dashboard.getId();

        return dashboardsDao.exists(projectId, dashboardId)
            .thenApply(exists -> {
                if (exists) {
                    String message = String.format(
                        "dashboard \"%s\" with version %s is out of date",
                        dashboardId,
                        dashboard.getVersion()
                    );
                    throw new ConflictException(message);
                }
                throw dashboardNotFound(dashboardId);
            });
    }

    @Override
    public CompletableFuture<Empty> deleteDashboard(DeleteDashboardRequest request, AuthSubject authSubject) {
        String dashboardId = request.getDashboardId();

        return dashboardsDao.readById(request.getDashboardId())
                .thenCompose(existingDashboardOpt -> {
                    if (existingDashboardOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(dashboardNotFound(dashboardId));
                    }

                    var existingDashboard = existingDashboardOpt.get();
                    var projectId = existingDashboard.getParentId();

                    return authorizer.authorize(authSubject, existingDashboard.getParentId(), Permission.CONFIGS_DELETE)
                            .thenCompose(account -> dashboardsDao.delete(projectId, dashboardId))
                            .thenApply(deleted -> {
                                if (!deleted) {
                                    throw dashboardNotFound(dashboardId);
                                }
                                return Empty.getDefaultInstance();
                            });
                });
    }

    private CompletableFuture<TokenBasePage<Dashboard>> withFavorites(String login, CompletableFuture<TokenBasePage<Entity>> dashboardEntitiesFuture) {
        return dashboardEntitiesFuture.thenCompose(pagedResult ->
                monitoringFavoritesDao.list(login, EntityType.ENTITY_TYPE_DASHBOARD)
                        .thenApply(favoriteRecords -> pagedResult.map(entity -> {
                            Dashboard dashboard = MonitoringDashboardConverter.decode(entity);
                            boolean isFavorite = favoriteRecords.stream().anyMatch(f -> f.getId().equals(dashboard.getId()));
                            return dashboard.toBuilder().setIsFavorite(isFavorite).build();
                        })));
    }

    private CompletableFuture<Optional<Dashboard>> withFavorite(String login, CompletableFuture<Optional<Entity>> dashboardEntityFuture) {
        return dashboardEntityFuture.thenCompose(dashboardEntityOpt -> {
            if (dashboardEntityOpt.isEmpty()) {
                return CompletableFuture.completedFuture(Optional.empty());
            }
            Dashboard dashboard = MonitoringDashboardConverter.decode(dashboardEntityOpt.get());

            return monitoringFavoritesDao.exists(new FavoriteRecord(login, EntityType.ENTITY_TYPE_DASHBOARD, dashboard.getId()))
                    .thenApply(exists -> {
                        Dashboard newDashboard = dashboard.toBuilder().setIsFavorite(exists).build();
                        return Optional.of(newDashboard);
                    });
        });
    }

    private static NotFoundException dashboardNotFound(String dashboardId) {
        return new NotFoundException(String.format("no dashboard with id %s found", dashboardId));
    }
}
