package ru.yandex.solomon.gateway.api.v3alpha;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.CreateFavoriteRequest;
import ru.yandex.monitoring.v3.DeleteFavoriteRequest;
import ru.yandex.monitoring.v3.Favorite;
import ru.yandex.monitoring.v3.ListFavoriteResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.MonitoringFavoritesDao;
import ru.yandex.solomon.conf.db3.ydb.FavoriteRecord;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class MonitoringFavoriteService {
    private final MonitoringFavoritesDao favoriteDao;
    private final MonitoringDashboardsDao monitoringDashboardsDao;
    private final ProjectsDao projectsDao;

    @Autowired
    public MonitoringFavoriteService(
        MonitoringFavoritesDao favoriteDao,
        MonitoringDashboardsDao monitoringDashboardsDao,
        ProjectsDao projectsDao)
    {
        this.favoriteDao = favoriteDao;
        this.monitoringDashboardsDao = monitoringDashboardsDao;
        this.projectsDao = projectsDao;
    }

    public CompletableFuture<ListFavoriteResponse> list(AuthSubject authSubject) {
        String login = authSubject.getUniqueId();
        return favoriteDao.listAll(login).thenCompose(this::mapRecordsToResponse);
    }

    public CompletableFuture<ListFavoriteResponse> list(EntityType entityType, AuthSubject authSubject) {
        String login = authSubject.getUniqueId();
        return favoriteDao.list(login, entityType).thenCompose(this::mapRecordsToResponse);
    }

    public CompletableFuture<ListFavoriteResponse> mapRecordsToResponse(List<FavoriteRecord> records) {
        var futures = records.stream()
                .map(record -> findFavoriteEntity(record.getType(), record.getId()))
                .collect(Collectors.toList());

        return CompletableFutures.allOf(futures).thenApply(favoriteOpts -> {
            var favorites = favoriteOpts.stream()
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .collect(Collectors.toList());
            return ListFavoriteResponse.newBuilder()
                    .addAllFavorites(favorites)
                    .build();
        });
    }

    public CompletableFuture<Favorite> create(CreateFavoriteRequest request, AuthSubject authSubject) {
        String login = authSubject.getUniqueId();

        EntityType entityType = toModel(request.getType());

        return findFavoriteEntity(entityType, request.getId())
                .thenCompose(favoriteOpt -> {
                    if (favoriteOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(new ConflictException("favorite entity doesn't exist"));
                    }
                    var record = new FavoriteRecord(login, entityType, request.getId());
                    return favoriteDao.create(record)
                            .thenApply(created -> {
                                if (!created) {
                                    throw new ConflictException("favorite already exists");
                                }
                                return favoriteOpt.get();
                            });
                });
    }

    private static EntityType toModel(ru.yandex.monitoring.v3.EntityType entityType) {
        return switch (entityType) {
            case ENTITY_TYPE_NOT_SPECIFIED, UNRECOGNIZED -> EntityType.ENTITY_TYPE_NOT_SPECIFIED;
            case ENTITY_TYPE_DASHBOARD -> EntityType.ENTITY_TYPE_DASHBOARD;
            case ENTITY_TYPE_PROJECT -> EntityType.ENTITY_TYPE_PROJECT;
        };
    }

    public CompletableFuture<Empty> delete(DeleteFavoriteRequest request, AuthSubject authSubject) {
        String login = authSubject.getUniqueId();

        EntityType entityType = toModel(request.getType());

        return findFavoriteEntity(entityType, request.getId())
                .thenCompose(favoriteOpt -> {
                    if (favoriteOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(new ConflictException("favorite entity doesn't exist"));
                    }
                    var record = new FavoriteRecord(login, entityType, request.getId());
                    return favoriteDao.delete(record)
                            .thenApply(deleted -> {
                                if (!deleted) {
                                    throw new NotFoundException("favorite doesn't exist");
                                }
                                return Empty.getDefaultInstance();
                            });
                });
    }

    private CompletableFuture<Optional<Favorite>> findFavoriteEntity(EntityType type, String id) {
        switch (type) {
            case ENTITY_TYPE_DASHBOARD:
                return monitoringDashboardsDao.readById(id).thenApply(dashboardOpt ->
                        dashboardOpt.map(dashboard -> Favorite.newBuilder()
                                .setParentPath(dashboard.getParentId())
                                .setId(dashboard.getId())
                                .setType(ru.yandex.monitoring.v3.EntityType.ENTITY_TYPE_DASHBOARD)
                                .setName(dashboard.getName())
                                .build()));
            case ENTITY_TYPE_PROJECT:
                return projectsDao.findById(id).thenApply(projectOpt ->
                        projectOpt.map(project -> Favorite.newBuilder()
                                .setId(project.getId())
                                .setType(ru.yandex.monitoring.v3.EntityType.ENTITY_TYPE_PROJECT)
                                .setName(project.getName())
                                .build()));
            default:
                throw new BadRequestException("unexpected type: " + type);
        }
    }
}
