package ru.yandex.solomon.gateway.api.v3alpha;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.v3.GetQuickLinksRequest;
import ru.yandex.monitoring.v3.QuickLinks;
import ru.yandex.monitoring.v3.QuickLinksData;
import ru.yandex.monitoring.v3.UpdateQuickLinksRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.conf.db3.ConfigV3DaoContext;
import ru.yandex.solomon.conf.db3.QuickLinksDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;

/**
 * @author Oleg Baryshnikov
 */
@Import(ConfigV3DaoContext.class)
@Component
@ParametersAreNonnullByDefault
public class MonitoringQuickLinksServiceImpl implements MonitoringQuickLinksService {
    private final ProjectsDao projectsDao;
    private final QuickLinksDao quickLinksDao;

    @Autowired
    public MonitoringQuickLinksServiceImpl(ProjectsDao projectsDao, QuickLinksDao quickLinksDao) {
        this.projectsDao = projectsDao;
        this.quickLinksDao = quickLinksDao;
    }

    @Override
    public CompletableFuture<QuickLinks> getQuickLinks(GetQuickLinksRequest request) {
        return quickLinksDao.read(request.getProjectId())
            .thenApply(quickLinks -> {
                if (quickLinks.isEmpty()) {
                    return QuickLinks.newBuilder()
                            .setProjectId(request.getProjectId())
                            .setData(QuickLinksData.getDefaultInstance())
                            .build();
                }
                return QuickLinksConverter.decode(quickLinks.get());
            });
    }

    @Override
    public CompletableFuture<QuickLinks> updateQuickLinks(UpdateQuickLinksRequest request, Account account) {
        Timestamp now = Timestamps.fromMillis(Instant.now().toEpochMilli());

        QuickLinksValidator.validate(request);

        String projectId = request.getProjectId();

        var proto = QuickLinks.newBuilder()
            .setProjectId(projectId)
            .setData(request.getData())
            .setUpdatedAt(now)
            .setUpdatedBy(account.getId())
            .setVersion(request.getVersion())
            .build();

        var model = QuickLinksConverter.encode(proto);

        return checkForeignRefs(projectId)
            .thenCompose(aVoid -> quickLinksDao.upsert(model))
            .thenApply(upsertedOpt -> {
                if (upsertedOpt.isEmpty()) {
                    throw new ConflictException(String.format("quick links for project %s with version %s is out of date", projectId, model.getVersion()));
                }
                return upsertedOpt.get();
            })
            .thenApply(QuickLinksConverter::decode);
    }

    private CompletableFuture<Void> checkForeignRefs(String projectId) {
        return projectsDao.exists(projectId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new NotFoundException(String.format("project %s does not exist", projectId));
                }
            });
    }
}
