package ru.yandex.solomon.gateway.api.v3alpha;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.v3.QuickLinksData;
import ru.yandex.solomon.conf.db3.QuickLinksRecord;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class QuickLinksConverter {
    public static QuickLinksRecord encode(ru.yandex.monitoring.v3.QuickLinks proto) {
        return QuickLinksRecord.newBuilder()
            .setProjectId(proto.getProjectId())
            .setItems(toLinkItems(proto.getData().getItemsList()))
            .setCreatedAtMillis(Timestamps.toMillis(proto.getCreatedAt()))
            .setUpdatedAtMillis(Timestamps.toMillis(proto.getUpdatedAt()))
            .setCreatedBy(proto.getCreatedBy())
            .setUpdatedBy(proto.getUpdatedBy())
            .setVersion((int) proto.getVersion())
            .build();
    }

    public static ru.yandex.monitoring.v3.QuickLinks decode(QuickLinksRecord model) {
        var data = QuickLinksData.newBuilder()
            .addAllItems(fromMenuItems(model.getItems()))
            .build();

        return ru.yandex.monitoring.v3.QuickLinks.newBuilder()
            .setProjectId(model.getProjectId())
            .setData(data)
            .setCreatedAt(Timestamps.fromMillis(model.getCreatedAtMillis()))
            .setUpdatedAt(Timestamps.fromMillis(model.getUpdatedAtMillis()))
            .setCreatedBy(model.getCreatedBy())
            .setUpdatedBy(model.getUpdatedBy())
            .setVersion(model.getVersion())
            .build();
    }

    private static QuickLinksRecord.LinkItem[] toLinkItems(List<QuickLinksData.Item> itemsList) {
        return itemsList.stream()
            .map(QuickLinksConverter::toMenuItem)
            .toArray(QuickLinksRecord.LinkItem[]::new);
    }

    private static QuickLinksRecord.LinkItem toMenuItem(QuickLinksData.Item proto) {
        switch (proto.getTypeCase()) {
            case LINK: {
                QuickLinksData.LinkItem item = proto.getLink();
                return new QuickLinksRecord.LinkItem(item.getTitle(), item.getUrl(), new QuickLinksRecord.LinkItem[0]);
            }
            case FOLDER: {
                QuickLinksData.FolderItem item = proto.getFolder();
                return new QuickLinksRecord.LinkItem(item.getTitle(), "", toLinkItems(item.getItemsList()));
            }
            case TYPE_NOT_SET:
                throw new BadRequestException("item type not set");
            default:
                throw new IllegalStateException("unknown item type: " + proto.getTypeCase());
        }
    }

    private static List<QuickLinksData.Item> fromMenuItems(QuickLinksRecord.LinkItem[] items) {
        return Arrays.stream(items)
            .map(QuickLinksConverter::fromMenuItem)
            .collect(Collectors.toList());
    }

    private static QuickLinksData.Item fromMenuItem(QuickLinksRecord.LinkItem item) {
        if (item.getUrl().isEmpty()) {
            return QuickLinksData.Item.newBuilder()
                .setFolder(QuickLinksData.FolderItem.newBuilder()
                    .setTitle(item.getTitle())
                    .addAllItems(fromMenuItems(item.getItems()))
                    .build())
                .build();
        }

        return QuickLinksData.Item.newBuilder()
            .setLink(QuickLinksData.LinkItem.newBuilder()
                .setTitle(item.getTitle())
                .setUrl(item.getUrl())
                .build())
            .build();
    }
}
