package ru.yandex.solomon.gateway.api.v3alpha.cloud;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.v3.cloud.CreateDashboardRequest;
import ru.yandex.monitoring.v3.cloud.Dashboard;
import ru.yandex.monitoring.v3.cloud.DeleteDashboardRequest;
import ru.yandex.monitoring.v3.cloud.GetDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListDashboardResponse;
import ru.yandex.monitoring.v3.cloud.UpdateDashboardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.conf.db3.CloudFavoritesDao;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.ydb.CloudFavoriteRecord;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.CloudDashboardConverter;
import ru.yandex.solomon.gateway.cloud.search.SearchEvent;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSink;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class CloudMonitoringDashboardServiceImpl implements CloudMonitoringDashboardService {
    private final String entityIdPrefix;
    private final CloudByFolderResolver cloudByFolderResolver;
    private final Authorizer authorizer;
    private final MonitoringDashboardsDao dashboardsDao;
    private final CloudFavoritesDao cloudFavoritesDao;
    private final SearchEventSink searchEventSink;

    public CloudMonitoringDashboardServiceImpl(
            String entityIdPrefix,
            CloudByFolderResolver cloudByFolderResolver,
            Authorizer authorizer,
            MonitoringDashboardsDao dashboardsDao,
            CloudFavoritesDao cloudFavoritesDao,
            SearchEventSink searchEventSink)
    {
        this.cloudByFolderResolver = cloudByFolderResolver;
        this.authorizer = authorizer;
        this.dashboardsDao = dashboardsDao;
        this.entityIdPrefix = entityIdPrefix;
        this.cloudFavoritesDao = cloudFavoritesDao;
        this.searchEventSink = searchEventSink;
    }

    @Override
    public CompletableFuture<Dashboard> getDashboard(GetDashboardRequest request, AuthSubject authSubject) {
        String dashboardId = request.getDashboardId();

        return withFavorite(authSubject.getUniqueId(), dashboardsDao.readById(dashboardId))
            .thenCompose(dashboardOpt -> {
                if (dashboardOpt.isEmpty()) {
                    return CompletableFuture.failedFuture(dashboardNotFound(dashboardId));
                }

                var dashboard = dashboardOpt.get();
                var folderId = dashboard.getFolderId();

                return cloudByFolderResolver.resolveCloudId(dashboard.getFolderId())
                        .thenCompose(cloudId -> authorizer.authorize(authSubject, cloudId, folderId, Permission.CONFIGS_GET))
                        .thenApply(account -> dashboard);
            });
    }

    @Override
    public CompletableFuture<ListDashboardResponse> listDashboards(
            ListDashboardRequest request,
            AuthSubject authSubject) {
        String folderId = request.getFolderId();
        if (folderId.isBlank()) {
            return CompletableFuture.failedFuture(new BadRequestException("folderId must be specified"));
        }

        return cloudByFolderResolver.resolveCloudId(folderId)
                .thenCompose(cloudId -> authorizer.authorize(authSubject, cloudId, folderId, Permission.CONFIGS_LIST)
                        .thenCompose(account -> withFavorites(authSubject.getUniqueId(), dashboardsDao.list(request.getFolderId(), request.getFilter(), (int) request.getPageSize(), request.getPageToken()))
                                .thenApply(pagedResult -> ListDashboardResponse.newBuilder()
                                        .setNextPageToken(pagedResult.getNextPageToken())
                                        .addAllDashboards(pagedResult.getItems())
                                        .build())));
    }

    @Override
    public CompletableFuture<Dashboard> createDashboard(CreateDashboardRequest request, AuthSubject authSubject) {
        String folderId = request.getFolderId();

        return cloudByFolderResolver.resolveCloudId(folderId)
                .thenCompose(cloudId ->
                        authorizer.authorize(authSubject, cloudId, folderId, Permission.CONFIGS_UPDATE)
                                .thenCompose(account -> createDashboardImpl(cloudId, folderId, request, authSubject)));
    }

    private CompletableFuture<Dashboard> createDashboardImpl(
            String cloudId,
            String folderId,
            CreateDashboardRequest request,
            AuthSubject authSubject) {
        Instant now = Instant.now();

        String dashboardId = IdGenerator.generateId(entityIdPrefix);

        Dashboard dashboard = Dashboard.newBuilder()
                .setFolderId(folderId)
                .setId(dashboardId)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllWidgets(request.getWidgetsList())
                .setParametrization(request.getParametrization())
                .setCreatedAt(Timestamps.fromMillis(now.toEpochMilli()))
                .setUpdatedAt(Timestamps.fromMillis(now.toEpochMilli()))
                .setCreatedBy(authSubject.getUniqueId())
                .setUpdatedBy(authSubject.getUniqueId())
                .setVersion(0)
                .build();

        Entity entity = CloudDashboardConverter.encode(dashboard, "");

        return dashboardsDao.insert(entity)
                .thenApply(inserted -> {
                    if (!inserted) {
                        throw new ConflictException("dashboard " + dashboardId + " already exists");
                    }
                    searchEventSink.accept(SearchEvent.dashboard(cloudId, entity));
                    return dashboard;
                });
    }

    @Override
    public CompletableFuture<Dashboard> updateDashboard(UpdateDashboardRequest request, AuthSubject authSubject) {
        var dashboardId = request.getDashboardId();

        return dashboardsDao.readById(dashboardId)
                .thenCompose(existingDashboardOpt -> {
                    if (existingDashboardOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(dashboardNotFound(dashboardId));
                    }

                    var existingDashboard = existingDashboardOpt.get();
                    String folderId = existingDashboard.getParentId();

                    return cloudByFolderResolver.resolveCloudId(folderId)
                            .thenCompose(cloudId -> authorizer.authorize(authSubject, cloudId, folderId, Permission.CONFIGS_UPDATE)
                                    .thenCompose(account -> {
                                        Timestamp now = Timestamps.fromMillis(Instant.now().toEpochMilli());

                                        Dashboard dashboard = Dashboard.newBuilder()
                                                .setFolderId(folderId)
                                                .setId(request.getDashboardId())
                                                .setName(request.getName())
                                                .setDescription(request.getDescription())
                                                .addAllWidgets(request.getWidgetsList())
                                                .setParametrization(request.getParametrization())
                                                .setUpdatedAt(now)
                                                .setUpdatedBy(authSubject.getUniqueId())
                                                .setVersion(request.getVersion())
                                                .setCreatedAt(Timestamps.fromMillis(existingDashboard.getCreatedAt()))
                                                .setCreatedBy(existingDashboard.getCreatedBy())
                                                .build();

                                        Entity entity = CloudDashboardConverter.encode(dashboard, existingDashboard.getLocalId());
                                        var updateFuture = dashboardsDao.update(entity).thenApply(opt -> {
                                            opt.ifPresent(updated -> searchEventSink.accept(SearchEvent.dashboard(cloudId, updated)));
                                            return opt;
                                        });

                                        return withFavorite(authSubject.getUniqueId(), updateFuture)
                                                .thenCompose(updatedDashboardOpt -> checkExistence(dashboard, updatedDashboardOpt));
                                    }));
                });
    }

    private CompletableFuture<Dashboard> checkExistence(
            Dashboard dashboard,
            Optional<Dashboard> updatedDashboardOpt) {
        if (updatedDashboardOpt.isPresent()) {
            return CompletableFuture.completedFuture(updatedDashboardOpt.get());
        }

        String folderId = dashboard.getFolderId();
        String dashboardId = dashboard.getId();

        return dashboardsDao.exists(folderId, dashboardId)
                .thenApply(exists -> {
                    if (exists) {
                        String message = String.format(
                                "dashboard %s with version %s is out of date",
                                dashboardId,
                                dashboard.getVersion()
                        );
                        throw new ConflictException(message);
                    }
                    throw dashboardNotFound(dashboardId);
                });
    }

    @Override
    public CompletableFuture<Empty> deleteDashboard(DeleteDashboardRequest request, AuthSubject authSubject) {
        String dashboardId = request.getDashboardId();

        long now = System.currentTimeMillis();
        return dashboardsDao.readById(dashboardId)
                .thenCompose(existingDashboardOpt -> {
                    if (existingDashboardOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(dashboardNotFound(dashboardId));
                    }

                    var existingDashboard = existingDashboardOpt.get();
                    var folderId = existingDashboard.getParentId();

                    return cloudByFolderResolver.resolveCloudId(folderId)
                            .thenCompose(cloudId ->
                                    authorizer.authorize(authSubject, cloudId, folderId, Permission.CONFIGS_DELETE)
                                            .thenCompose(account -> dashboardsDao.delete(folderId, dashboardId))
                                            .thenApply(deleted -> {
                                                if (!deleted) {
                                                    throw dashboardNotFound(dashboardId);
                                                }
                                                searchEventSink.accept(SearchEvent.dashboard(cloudId, existingDashboard).deletedAt(now));
                                                return Empty.getDefaultInstance();
                                            }));
                });
    }

    private CompletableFuture<TokenBasePage<Dashboard>> withFavorites(String login, CompletableFuture<TokenBasePage<Entity>> dashboardEntitiesFuture) {
        return dashboardEntitiesFuture.thenCompose(pagedResult ->
                cloudFavoritesDao.list(login, EntityType.ENTITY_TYPE_DASHBOARD)
                        .thenApply(favoriteRecords -> pagedResult.map(entity -> {
                            Dashboard dashboard = CloudDashboardConverter.decode(entity);
                            boolean isFavorite = favoriteRecords.stream().anyMatch(f -> f.getId().equals(dashboard.getId()));
                            return dashboard.toBuilder().setIsFavorite(isFavorite).build();
                        })));
    }

    private CompletableFuture<Optional<Dashboard>> withFavorite(String login, CompletableFuture<Optional<Entity>> dashboardEntityFuture) {
        return dashboardEntityFuture.thenCompose(dashboardEntityOpt -> {
            if (dashboardEntityOpt.isEmpty()) {
                return CompletableFuture.completedFuture(Optional.empty());
            }
            Dashboard dashboard = CloudDashboardConverter.decode(dashboardEntityOpt.get());

            return cloudFavoritesDao.exists(new CloudFavoriteRecord(login, EntityType.ENTITY_TYPE_DASHBOARD, dashboard.getId()))
                    .thenApply(exists -> {
                        Dashboard newDashboard = dashboard.toBuilder().setIsFavorite(exists).build();
                        return Optional.of(newDashboard);
                    });
        });
    }

    private static NotFoundException dashboardNotFound(String dashboardId) {
        return new NotFoundException(String.format("no dashboard with id %s found", dashboardId));
    }
}
