package ru.yandex.solomon.gateway.api.v3alpha.cloud;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.cloud.CreateFavoriteRequest;
import ru.yandex.monitoring.v3.cloud.DeleteFavoriteRequest;
import ru.yandex.monitoring.v3.cloud.Favorite;
import ru.yandex.monitoring.v3.cloud.FavoriteServiceGrpc;
import ru.yandex.monitoring.v3.cloud.ListFavoriteRequest;
import ru.yandex.monitoring.v3.cloud.ListFavoriteResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.conf.db3.CloudFavoritesDao;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardsDao;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.ydb.CloudFavoriteRecord;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcCloudFavoriteService extends FavoriteServiceGrpc.FavoriteServiceImplBase implements GrpcService {
    private final CloudFavoritesDao favoriteDao;
    private final MonitoringDashboardsDao dashboardsDao;
    private final CloudServiceDashboardsDao cloudServiceDashboardsDao;

    @Autowired
    public GrpcCloudFavoriteService(
        CloudFavoritesDao cloudFavoritesDao,
        MonitoringDashboardsDao dashboardsDao,
        CloudServiceDashboardsDao cloudServiceDashboardsDao)
    {
        this.favoriteDao = cloudFavoritesDao;
        this.dashboardsDao = dashboardsDao;
        this.cloudServiceDashboardsDao = cloudServiceDashboardsDao;
    }

    @Override
    public void list(ListFavoriteRequest request, StreamObserver<ListFavoriteResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(this::listImpl)
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ListFavoriteResponse> listImpl() {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());

        String login = authSubject.getUniqueId();

        return favoriteDao.listAll(login).thenCompose(records -> {
            var futures = records.stream()
                    .map(record -> findFavoriteEntity(record.getType(), record.getId()))
                    .collect(Collectors.toList());

            return CompletableFutures.allOf(futures).thenApply(favoriteOpts -> {
                var favorites = favoriteOpts.stream()
                        .filter(Optional::isPresent)
                        .map(Optional::get)
                        .collect(Collectors.toList());
                return ListFavoriteResponse.newBuilder()
                        .addAllFavorites(favorites)
                        .build();
            });
        });
    }

    @Override
    public void create(CreateFavoriteRequest request, StreamObserver<Favorite> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> createImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Favorite> createImpl(CreateFavoriteRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        String login = authSubject.getUniqueId();

        EntityType entityType = toModel(request.getType());
        return findFavoriteEntity(entityType, request.getId())
                .thenCompose(favoriteOpt -> {
                    if (favoriteOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(new ConflictException("favorite entity doesn't exist"));
                    }
                    var record = new CloudFavoriteRecord(login, entityType, request.getId());
                    return favoriteDao.create(record)
                            .thenApply(created -> {
                                if (!created) {
                                    throw new ConflictException("favorite already exists");
                                }
                                return favoriteOpt.get();
                            });
                });
    }

    @Override
    public void delete(DeleteFavoriteRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> deleteImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> deleteImpl(DeleteFavoriteRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        String login = authSubject.getUniqueId();

        EntityType entityType = toModel(request.getType());

        return findFavoriteEntity(entityType, request.getId())
                .thenCompose(favoriteOpt -> {
                    if (favoriteOpt.isEmpty()) {
                        return CompletableFuture.failedFuture(new ConflictException("favorite entity doesn't exist"));
                    }
                    var record = new CloudFavoriteRecord(login, entityType, request.getId());
                    return favoriteDao.delete(record)
                            .thenApply(deleted -> {
                                if (!deleted) {
                                    throw new NotFoundException("favorite doesn't exist");
                                }
                                return Empty.getDefaultInstance();
                            });
                });
    }

    private CompletableFuture<Optional<Favorite>> findFavoriteEntity(EntityType type, String id) {
        switch (type) {
            case ENTITY_TYPE_DASHBOARD:
                return dashboardsDao.readById(id).thenApply(dashboardOpt ->
                        dashboardOpt.map(dashboard -> Favorite.newBuilder()
                                .setParentPath(dashboard.getParentId())
                                .setId(dashboard.getId())
                                .setType(ru.yandex.monitoring.v3.cloud.EntityType.ENTITY_TYPE_DASHBOARD)
                                .setName(dashboard.getName())
                                .build()));
            case ENTITY_TYPE_SERVICE_DASHBOARD:
                return cloudServiceDashboardsDao.read(id).thenApply(serviceDashboardOpt ->
                        serviceDashboardOpt.map(dashboard -> Favorite.newBuilder()
                                .setParentPath(dashboard.getService())
                                .setId(dashboard.getId())
                                .setType(ru.yandex.monitoring.v3.cloud.EntityType.ENTITY_TYPE_SERVICE_DASHBOARD)
                                .setName(dashboard.getName())
                                .build()));
            default:
                throw new BadRequestException("unexpected type: " + type);
        }
    }

    private static EntityType toModel(ru.yandex.monitoring.v3.cloud.EntityType entityType) {
        return switch (entityType) {
            case ENTITY_TYPE_NOT_SPECIFIED, UNRECOGNIZED -> EntityType.ENTITY_TYPE_NOT_SPECIFIED;
            case ENTITY_TYPE_DASHBOARD -> EntityType.ENTITY_TYPE_DASHBOARD;
            case ENTITY_TYPE_SERVICE_DASHBOARD -> EntityType.ENTITY_TYPE_SERVICE_DASHBOARD;
        };
    }
}
