package ru.yandex.solomon.gateway.api.v3alpha.cloud;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.cloud.CreateDashboardRequest;
import ru.yandex.monitoring.v3.cloud.Dashboard;
import ru.yandex.monitoring.v3.cloud.DashboardServiceGrpc;
import ru.yandex.monitoring.v3.cloud.DeleteDashboardRequest;
import ru.yandex.monitoring.v3.cloud.GetDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListDashboardResponse;
import ru.yandex.monitoring.v3.cloud.UpdateDashboardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * Cloud dashboard gRPC service implementation
 *
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcCloudMonitoringDashboardService extends DashboardServiceGrpc.DashboardServiceImplBase implements GrpcService {
    private final CloudMonitoringDashboardService service;

    @Autowired
    public GrpcCloudMonitoringDashboardService(CloudMonitoringDashboardService service) {
        this.service = service;
    }

    @Override
    public void get(GetDashboardRequest request, StreamObserver<Dashboard> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Dashboard> getImpl(GetDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return service.getDashboard(request, authSubject);
    }

    @Override
    public void list(ListDashboardRequest request, StreamObserver<ListDashboardResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> listImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<ListDashboardResponse> listImpl(ListDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return service.listDashboards(request, authSubject);
    }

    @Override
    public void create(CreateDashboardRequest request, StreamObserver<Dashboard> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> createImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Dashboard> createImpl(CreateDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return service.createDashboard(request, authSubject);
    }

    @Override
    public void update(UpdateDashboardRequest request, StreamObserver<Dashboard> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> updateImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Dashboard> updateImpl(UpdateDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return service.updateDashboard(request, authSubject);
    }

    @Override
    public void delete(DeleteDashboardRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> deleteImpl(request))
                .exceptionally(throwable -> {
                    throw GrpcApiExceptionResolver.doResolveException(throwable);
                })
            , responseObserver);
    }

    private CompletableFuture<Empty> deleteImpl(DeleteDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return service.deleteDashboard(request, authSubject);
    }
}
