package ru.yandex.solomon.gateway.api.v3alpha.cloud;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.cloud.GetServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ListServiceDashboardResponse;
import ru.yandex.monitoring.v3.cloud.ServiceDashboard;
import ru.yandex.monitoring.v3.cloud.ServiceDashboardServiceGrpc;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardsDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.CloudServiceDashboardConverter;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcServiceDashboardService extends ServiceDashboardServiceGrpc.ServiceDashboardServiceImplBase implements GrpcService {
    private final Optional<CloudByFolderResolver> cloudByFolderResolver;
    private final Authorizer authorizer;
    private final CloudServiceDashboardsDao serviceDashboardsDao;
    private final ShardsDao shardsDao;

    @Autowired
    public GrpcServiceDashboardService(
            Optional<CloudByFolderResolver> cloudByFolderResolver,
            Authorizer authorizer,
            CloudServiceDashboardsDao serviceDashboardsDao,
            ShardsDao shardsDao)
    {
        this.cloudByFolderResolver = cloudByFolderResolver;
        this.authorizer = authorizer;
        this.serviceDashboardsDao = serviceDashboardsDao;
        this.shardsDao = shardsDao;
    }

    @Override
    public void get(GetServiceDashboardRequest request, StreamObserver<ServiceDashboard> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ServiceDashboard> getImpl(GetServiceDashboardRequest request) {
        if (cloudByFolderResolver.isEmpty()) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException("not supported"));
        }

        String serviceDashboardId = request.getServiceDashboardId();

        return serviceDashboardsDao.read(serviceDashboardId)
                .thenApply(serviceDashboardOpt -> {
                    if (serviceDashboardOpt.isEmpty()) {
                        throw serviceDashboardNotFound(serviceDashboardId);
                    }

                    return CloudServiceDashboardConverter.decode(serviceDashboardOpt.get());
                });
    }

    @Override
    public void list(ListServiceDashboardRequest request, StreamObserver<ListServiceDashboardResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> listImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ListServiceDashboardResponse> listImpl(ListServiceDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());

        if (cloudByFolderResolver.isEmpty()) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException("not supported"));
        }

        return cloudByFolderResolver.get().resolveCloudId(request.getFolderId())
                .thenCompose(cloudId -> authorizer.authorize(authSubject, cloudId, request.getFolderId(), Permission.CONFIGS_GET).thenCompose(account -> {
                return getRelatedServices(cloudId, request.getFolderId())
                    .thenCompose(services -> {
                        if (services.isEmpty()) {
                            return CompletableFuture.completedFuture(ListServiceDashboardResponse.getDefaultInstance());
                        }

                        return serviceDashboardsDao.list(services, request.getFilter(), (int) request.getPageSize(), request.getPageToken())
                                .thenApply(pagedResult -> ListServiceDashboardResponse.newBuilder()
                                        .setNextPageToken(pagedResult.getNextPageToken())
                                        .addAllServiceDashboards(pagedResult.getItems().stream()
                                                .map(CloudServiceDashboardConverter::decode)
                                                .collect(Collectors.toList()))
                                        .build());
                    });
        }));
    }

    private CompletableFuture<Set<String>> getRelatedServices(String cloudId, String folderId) {
        String clusterId = cloudId + "_" + folderId;
        return shardsDao.findByClusterId(cloudId, "", clusterId)
                .thenApply(shards -> shards.stream()
                        .map(Shard::getServiceName)
                        .collect(Collectors.toSet()));
    }

    private NotFoundException serviceDashboardNotFound(String dashboardId) {
        return new NotFoundException(String.format("dashboard %s not found", dashboardId));
    }
}
