package ru.yandex.solomon.gateway.api.v3alpha.cloud.priv;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.v3.cloud.Dashboard;
import ru.yandex.monitoring.v3.cloud.ServiceDashboard;
import ru.yandex.monitoring.v3.cloud.priv.CreateServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.DeleteServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.GetServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ListServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ListServiceDashboardResponse;
import ru.yandex.monitoring.v3.cloud.priv.SyncServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.UpdateServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ValidateServiceDashboardRequest;
import ru.yandex.monitoring.v3.cloud.priv.ValidationServiceDashboardResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardRecord;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardsDao;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.CloudDashboardConverter;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.CloudServiceDashboardConverter;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class PrivServiceDashboardServiceImpl implements PrivServiceDashboardService {
    private final boolean production;
    private final String entityIdPrefix;
    private final InternalAuthorizer internalAuthorizer;
    private final CloudServiceDashboardsDao serviceDashboardsDao;
    private final MonitoringDashboardsDao dashboardsDao;
    private final ServiceProvidersDao serviceProvidersDao;
    private final ServiceDashboardValidationService serviceDashboardValidationService;

    public PrivServiceDashboardServiceImpl(
            boolean production,
            String entityIdPrefix,
            InternalAuthorizer internalAuthorizer,
            CloudServiceDashboardsDao serviceDashboardsDao,
            MonitoringDashboardsDao dashboardsDao,
            ServiceProvidersDao serviceProvidersDao,
            ServiceDashboardValidationService serviceDashboardValidationService)
    {
        this.production = production;
        this.entityIdPrefix = entityIdPrefix;
        this.internalAuthorizer = internalAuthorizer;
        this.serviceDashboardsDao = serviceDashboardsDao;
        this.dashboardsDao = dashboardsDao;
        this.serviceProvidersDao = serviceProvidersDao;
        this.serviceDashboardValidationService = serviceDashboardValidationService;
    }

    @Override
    public CompletableFuture<ServiceDashboard> get(GetServiceDashboardRequest request, AuthSubject authSubject) {
        ServiceDashboardRequestValidator.validate(request);
        String serviceDashboardId = request.getServiceDashboardId();
        return readServiceDashboard(serviceDashboardId);
    }

    @Override
    public CompletableFuture<ListServiceDashboardResponse> list(ListServiceDashboardRequest request, AuthSubject authSubject) {
        ServiceDashboardRequestValidator.validate(request);
        return serviceProvidersDao.read(request.getService()).thenCompose(serviceProviderOpt -> {
            if (serviceProviderOpt.isEmpty()) {
                throw new NotFoundException("service provider " + request.getService() + " not found");
            }
            return serviceDashboardsDao.list(Set.of(request.getService()), request.getFilter(), (int) request.getPageSize(), request.getPageToken())
                    .thenApply(pagedResult -> ListServiceDashboardResponse.newBuilder()
                            .setNextPageToken(pagedResult.getNextPageToken())
                            .addAllServiceDashboards(pagedResult.getItems().stream()
                                .map(CloudServiceDashboardConverter::decode)
                                .collect(Collectors.toList()))
                            .build());
        });
    }

    @Override
    public CompletableFuture<ServiceDashboard> create(CreateServiceDashboardRequest request, AuthSubject authSubject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());
        String login = authSubject.getUniqueId();

        ServiceDashboardRequestValidator.validate(request);

        String dashboardId = IdGenerator.generateId(entityIdPrefix);

        ServiceDashboard serviceDashboard = ServiceDashboard.newBuilder()
                .setId(dashboardId)
                .setService(request.getService())
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllWidgets(request.getWidgetsList())
                .setParametrization(request.getParametrization())
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(login)
                .setUpdatedBy(login)
                .addAllOwners(request.getOwnersList())
                .setSourceDashboardLink(request.getSourceDashboardLink())
                .build();

        return authorizeToModify(authSubject).thenCompose(aVoid ->
                serviceProvidersDao.read(request.getService()).thenCompose(serviceProviderOpt -> {
                    if (serviceProviderOpt.isEmpty()) {
                        throw new NotFoundException("service provider " + request.getService() + " not found");
                    }

                    ServiceProvider serviceProvider = serviceProviderOpt.get();
                    String cloudId = serviceProvider.getCloudId();
                    if (StringUtils.isBlank(cloudId)) {
                        throw new BadRequestException("cannot create service dashboard in internal service provider " + request.getService());
                    }

                    CloudServiceDashboardRecord entity = CloudServiceDashboardConverter.encode(serviceDashboard);

                    return serviceDashboardsDao.insert(entity).thenApply(inserted -> {
                        if (!inserted) {
                            throw new BadRequestException("service dashboard with id " + dashboardId + " already exists");
                        }
                        return serviceDashboard;
                    });
                }));
    }

    @Override
    public CompletableFuture<ServiceDashboard> update(UpdateServiceDashboardRequest request, AuthSubject authSubject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());
        String login = authSubject.getUniqueId();

        return authorizeToModify(authSubject).thenCompose(aVoid ->
                readServiceDashboard(request.getServiceDashboardId())
                        .thenCompose(existingServiceDashboard -> {
                            String service = existingServiceDashboard.getService();

                            ServiceDashboard serviceDashboard = ServiceDashboard.newBuilder()
                                    .setId(request.getServiceDashboardId())
                                    .setService(service)
                                    .addAllOwners(request.getOwnersList())
                                    .setSourceDashboardLink(request.getSourceDashboardLink())
                                    .setName(request.getName())
                                    .setDescription(request.getDescription())
                                    .addAllWidgets(request.getWidgetsList())
                                    .setParametrization(request.getParametrization())
                                    .setUpdatedAt(now)
                                    .setUpdatedBy(login)
                                    .setVersion(request.getVersion())
                                    .build();

                            CloudServiceDashboardRecord entity =
                                    CloudServiceDashboardConverter.encode(serviceDashboard);

                            return serviceDashboardsDao.update(entity)
                                    .thenCompose(updatedOpt -> {
                                        //noinspection OptionalIsPresent
                                        if (updatedOpt.isPresent()) {
                                            return CompletableFuture.completedFuture(CloudServiceDashboardConverter.decode(updatedOpt.get()));
                                        }

                                        return serviceDashboardsDao.exists(serviceDashboard.getId())
                                                .thenApply(exists -> {
                                                    if (exists) {
                                                        String message = String.format(
                                                                "service dashboard \"%s\" with version %s is out of date",
                                                                serviceDashboard.getId(),
                                                                serviceDashboard.getVersion()
                                                        );
                                                        throw new ConflictException(message);
                                                    }
                                                    throw serviceDashboardNotFound(serviceDashboard.getId());
                                                });
                                    });
                        }));
    }

    @Override
    public CompletableFuture<ValidationServiceDashboardResponse> validate(
            ValidateServiceDashboardRequest request,
            AuthSubject authSubject)
    {
        ServiceDashboard serviceDashboard = ServiceDashboard.newBuilder()
                .setService(request.getService())
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllWidgets(request.getWidgetsList())
                .setParametrization(request.getParametrization())
                .build();
        return serviceDashboardValidationService.validate(serviceDashboard);
    }

    @Override
    public CompletableFuture<ValidationServiceDashboardResponse> validateSync(
            SyncServiceDashboardRequest request,
            AuthSubject authSubject)
    {
        return authorizeToModify(authSubject).thenCompose(aVoid ->
                readServiceDashboard(request.getServiceDashboardId())
                        .thenCompose(serviceDashboard -> readCloudDashboard(request.getDashboardId())
                                .thenCompose(dashboard -> {
                                    var newServiceDashboard = ServiceDashboardSync.merge(dashboard, serviceDashboard, "");
                                    return serviceDashboardValidationService.validate(newServiceDashboard);
                                })));
    }

    @Override
    public CompletableFuture<ServiceDashboard> sync(SyncServiceDashboardRequest request, AuthSubject authSubject) {
        return authorizeToModify(authSubject).thenCompose(aVoid ->
                readServiceDashboard(request.getServiceDashboardId())
                        .thenCompose(serviceDashboard -> readCloudDashboard(request.getDashboardId())
                                .thenCompose(dashboard -> {
                                    var link = StringUtils.replace(request.getSourceDashboardLink(), "{folderId}", dashboard.getFolderId());
                                    var newServiceDashboard = ServiceDashboardSync.merge(dashboard, serviceDashboard, link);
                                    CloudServiceDashboardRecord entity = CloudServiceDashboardConverter.encode(newServiceDashboard);
                                    return serviceDashboardsDao.update(entity)
                                            .thenApply(updatedOpt -> {
                                                if (updatedOpt.isEmpty()) {
                                                    throw new ConflictException("service dashboard version conflict or " +
                                                            "service " +
                                                            "dashboard not found. Id: " + entity.getId() + ", version: " + entity.getVersion());
                                                }

                                                return CloudServiceDashboardConverter.decode(updatedOpt.get());
                                            });
                                })));
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteServiceDashboardRequest request, AuthSubject authSubject) {
        return authorizeToModify(authSubject).thenCompose(aVoid ->
                readServiceDashboard(request.getServiceDashboardId())
                        .thenCompose(account -> serviceDashboardsDao.delete(request.getServiceDashboardId()))
                        .thenApply(deleted -> {
                            if (!deleted) {
                                throw serviceDashboardNotFound(request.getServiceDashboardId());
                            }
                            return Empty.getDefaultInstance();
                        }));
    }

    private CompletableFuture<Void> authorizeToModify(AuthSubject authSubject) {
        if (production) {
            return internalAuthorizer.authorize(authSubject).thenApply(account -> null);
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<ServiceDashboard> readServiceDashboard(String serviceDashboardId) {
        return serviceDashboardsDao.read(serviceDashboardId)
                .thenApply(serviceDashboardOpt -> {
                    if (serviceDashboardOpt.isEmpty()) {
                        throw serviceDashboardNotFound(serviceDashboardId);
                    }

                    return CloudServiceDashboardConverter.decode(serviceDashboardOpt.get());
                });
    }

    private CompletableFuture<Dashboard> readCloudDashboard(String id) {
        return dashboardsDao.readById(id).thenApply(dashboardOpt -> {
            if (dashboardOpt.isEmpty()) {
                throw new NotFoundException("dashboard " + id + " not found");
            }
            return CloudDashboardConverter.decode(dashboardOpt.get());
        });
    }

    private NotFoundException serviceDashboardNotFound(String dashboardId) {
        return new NotFoundException(String.format("service dashboard %s not found", dashboardId));
    }
}
