package ru.yandex.solomon.gateway.api.v3alpha.cloud.priv.validators;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import ru.yandex.monitoring.v3.cloud.ChartWidget;
import ru.yandex.monitoring.v3.cloud.ServiceDashboard;
import ru.yandex.monitoring.v3.cloud.Widget;
import ru.yandex.monitoring.v3.cloud.priv.ServiceDashboardValidationIssue;
import ru.yandex.monitoring.v3.cloud.priv.ServiceDashboardValidationIssue.Severity;
import ru.yandex.solomon.gateway.api.v3alpha.cloud.priv.ServiceDashboardValidator;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Alexey Trushkin
 */
public class ParametersValidator implements ServiceDashboardValidator {

    private static final String SERVICE_KEY = "service";
    private static final String PARAMETRIZATION_ERROR = "Parametrization should contain at least one parameter";
    private static final String SELECTOR_BLANK_ERROR = "Selector should be specified";
    private static final String SELECTOR_KEY_ABSENT_ERROR = "Parametrization selector should contain label '%s'";
    private static final String SELECTOR_KEY_VALUE_NOT_EQUAL = "Label '%s' value should be '%s' in parametrization selector";
    private static final String UNIT_NOT_SPECIFIED_ERROR = "UnitFormat should be specified instead of '%s'";

    @Override
    public CompletableFuture<Collection<ServiceDashboardValidationIssue>> validate(ServiceDashboard serviceDashboard) {
        List<ServiceDashboardValidationIssue> result = new ArrayList<>();
        validateParameters(result, serviceDashboard);
        validateSelectors(result, serviceDashboard);
        for (Widget widget : serviceDashboard.getWidgetsList()) {
            validateWidget(result, widget);
        }
        return CompletableFuture.completedFuture(result);
    }

    private void validateParameters(List<ServiceDashboardValidationIssue> result, ServiceDashboard serviceDashboard) {
        if (serviceDashboard.getParametrization().getParametersCount() == 0) {
            result.add(createResult(Severity.WARN, PARAMETRIZATION_ERROR));
        }
    }

    private void validateWidget(List<ServiceDashboardValidationIssue> result, Widget widget) {
        if (widget.hasChart() && widget.getChart().hasVisualizationSettings()) {
            var visualizationSettings = widget.getChart().getVisualizationSettings();
            var validateLeft = false;
            var validateRight = false;
            for (ChartWidget.SeriesOverrides seriesOverrides : widget.getChart().getSeriesOverridesList()) {
                if (seriesOverrides.hasSettings()) {
                    if (seriesOverrides.getSettings().getYaxisPosition() == ChartWidget.SeriesOverrides.YaxisPosition.YAXIS_POSITION_LEFT) {
                        validateLeft = true;
                    }
                    if (seriesOverrides.getSettings().getYaxisPosition() == ChartWidget.SeriesOverrides.YaxisPosition.YAXIS_POSITION_RIGHT) {
                        validateRight = true;
                    }
                }
            }
            if (visualizationSettings.hasYaxisSettings()) {
                if (validateLeft && visualizationSettings.getYaxisSettings().hasLeft()) {
                    validateUnitFormat(visualizationSettings.getYaxisSettings().getLeft().getUnitFormat(), result);
                }
                if (validateRight && visualizationSettings.getYaxisSettings().hasRight()) {
                    validateUnitFormat(visualizationSettings.getYaxisSettings().getRight().getUnitFormat(), result);
                }
            }
        }
    }

    private void validateUnitFormat(ChartWidget.VisualizationSettings.UnitFormat unitFormat, List<ServiceDashboardValidationIssue> result) {
        if (unitFormat == ChartWidget.VisualizationSettings.UnitFormat.UNIT_FORMAT_NOT_SPECIFIED
                || unitFormat == ChartWidget.VisualizationSettings.UnitFormat.UNIT_NONE) {
            result.add(createResult(Severity.WARN, String.format(UNIT_NOT_SPECIFIED_ERROR, unitFormat)));
        }
    }

    private void validateSelectors(List<ServiceDashboardValidationIssue> result, ServiceDashboard serviceDashboard) {
        if (serviceDashboard.getParametrization().getSelectors().isBlank()) {
            result.add(createResult(Severity.WARN, SELECTOR_BLANK_ERROR));
            return;
        }
        Selectors parse = Selectors.parse(serviceDashboard.getParametrization().getSelectors());
        Selector serviceSelector = parse.findByKey(SERVICE_KEY);
        if (serviceSelector == null) {
            result.add(createResult(Severity.WARN, String.format(SELECTOR_KEY_ABSENT_ERROR, SERVICE_KEY)));
        } else if (!serviceSelector.getValue().equals(serviceDashboard.getService())) {
            result.add(createResult(Severity.ERROR, String.format(SELECTOR_KEY_VALUE_NOT_EQUAL, SERVICE_KEY, serviceDashboard.getService())));
        }
    }

    private ServiceDashboardValidationIssue createResult(Severity severity, String description) {
        return ServiceDashboardValidationIssue.newBuilder()
                .setSeverity(severity)
                .setDescription(description)
                .build();
    }
}
