package ru.yandex.solomon.gateway.api.v3alpha.dao.ydb;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import com.google.protobuf.util.Timestamps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monitoring.v3.cloud.ServiceDashboard;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardRecord;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class CloudServiceDashboardConverter {
    private static final Logger logger = LoggerFactory.getLogger(CloudServiceDashboardConverter.class);

    public static CloudServiceDashboardRecord encode(ServiceDashboard dashboard) {
        final String data;

        data = toDataJson(dashboard);

        return CloudServiceDashboardRecord.newBuilder()
                .setId(dashboard.getId())
                .setService(dashboard.getService())
                .setName(dashboard.getName())
                .setDescription(dashboard.getDescription())
                .setData(data)
                .setCreatedAt(Timestamps.toMillis(dashboard.getCreatedAt()))
                .setUpdatedAt(Timestamps.toMillis(dashboard.getUpdatedAt()))
                .setCreatedBy(dashboard.getCreatedBy())
                .setUpdatedBy(dashboard.getUpdatedBy())
                .setVersion((int) dashboard.getVersion())
                .build();
    }

    public static ServiceDashboard decode(CloudServiceDashboardRecord entity) {
        return fromDataJson(entity.getData())
                .setService(entity.getService())
                .setId(entity.getId())
                .setName(entity.getName())
                .setDescription(entity.getDescription())
                .setCreatedAt(Timestamps.fromMillis(entity.getCreatedAt()))
                .setUpdatedAt(Timestamps.fromMillis(entity.getUpdatedAt()))
                .setCreatedBy(entity.getCreatedBy())
                .setUpdatedBy(entity.getUpdatedBy())
                .setVersion(entity.getVersion())
                .build();
    }

    private static String toDataJson(ServiceDashboard dashboard) {
        var dataDashboard = ServiceDashboard.newBuilder()
                .addAllWidgets(dashboard.getWidgetsList())
                .setParametrization(dashboard.getParametrization())
                .setSourceDashboardLink(dashboard.getSourceDashboardLink())
                .addAllOwners(dashboard.getOwnersList())
                .build();

        try {
            return JsonFormat.printer().print(dataDashboard);
        } catch (Exception e) {
            String message = "failed to format dashboard data";
            logger.error(message, e);
            throw new IllegalStateException(message);
        }
    }

    private static ServiceDashboard.Builder fromDataJson(String json) {
        ServiceDashboard.Builder builder = ServiceDashboard.newBuilder();
        try {
            JsonFormat.parser().ignoringUnknownFields().merge(json, builder);
        } catch (InvalidProtocolBufferException e) {
            String message = "failed to parse json for dashboard data";
            logger.error(message, e);
            throw new IllegalStateException(message);
        }
        return builder;
    }
}
