package ru.yandex.solomon.gateway.api.v3alpha.priv;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.priv.AdminDashboardServiceGrpc;
import ru.yandex.monitoring.v3.priv.AdminDeleteByProjectIdDashboardRequest;
import ru.yandex.monitoring.v3.priv.AdminListAllDashboardRequest;
import ru.yandex.monitoring.v3.priv.AdminListAllDashboardResponse;
import ru.yandex.monitoring.v3.priv.AdminUpsertDashboardRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.MonitoringDashboardConverter;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcAdminDashboardService extends AdminDashboardServiceGrpc.AdminDashboardServiceImplBase implements GrpcService {
    private final InternalAuthorizer authorizer;
    private final MonitoringDashboardsDao dashboardsDao;

    @Autowired
    public GrpcAdminDashboardService(
            InternalAuthorizer authorizer,
            MonitoringDashboardsDao dashboardsDao)
    {
        this.authorizer = authorizer;
        this.dashboardsDao = dashboardsDao;
    }

    @Override
    public void listAll(AdminListAllDashboardRequest request, StreamObserver<AdminListAllDashboardResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> listAllImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<AdminListAllDashboardResponse> listAllImpl(AdminListAllDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return authorizer.authorize(authSubject)
                .thenCompose(account -> dashboardsDao.listAll(request.getFilter(), (int) request.getPageSize(), request.getPageToken()))
                .thenApply(dashboardTokenBasePage -> AdminListAllDashboardResponse.newBuilder()
                        .addAllDashboards(dashboardTokenBasePage.getItems().stream()
                                .map(MonitoringDashboardConverter::decode)
                                .collect(Collectors.toList()))
                        .setNextPageToken(dashboardTokenBasePage.getNextPageToken())
                        .build());
    }

    @Override
    public void upsert(AdminUpsertDashboardRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> upsertImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> upsertImpl(AdminUpsertDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return authorizer.authorize(authSubject)
                .thenCompose(account -> dashboardsDao.upsert(MonitoringDashboardConverter.encode(request.getDashboard(), "")))
                .thenApply(dashboardTokenBasePage -> Empty.getDefaultInstance());
    }

    @Override
    public void deleteByProjectId(AdminDeleteByProjectIdDashboardRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> deleteByProjectIdImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> deleteByProjectIdImpl(AdminDeleteByProjectIdDashboardRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return authorizer.authorize(authSubject)
                .thenCompose(account -> dashboardsDao.deleteByParentId(request.getProjectId()))
                .thenApply(dashboardTokenBasePage -> Empty.getDefaultInstance());
    }
}
