package ru.yandex.solomon.gateway.api.v3alpha.priv;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.priv.CreateServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.DeleteServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.GetServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderResponse;
import ru.yandex.monitoring.v3.priv.ServiceProvider;
import ru.yandex.monitoring.v3.priv.ServiceProviderServiceGrpc;
import ru.yandex.monitoring.v3.priv.UpdateServiceProviderRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.exception.handlers.GrpcApiExceptionResolver;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcServiceProviderService extends ServiceProviderServiceGrpc.ServiceProviderServiceImplBase implements GrpcService {
    private final ServiceProviderService serviceProviderService;

    @Autowired
    public GrpcServiceProviderService(ServiceProviderService serviceProviderService) {
        this.serviceProviderService = serviceProviderService;
    }

    @Override
    public void get(GetServiceProviderRequest request, StreamObserver<ServiceProvider> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> getImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ServiceProvider> getImpl(GetServiceProviderRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return serviceProviderService.get(request, authSubject);
    }

    @Override
    public void list(ListServiceProviderRequest request, StreamObserver<ListServiceProviderResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> listImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ListServiceProviderResponse> listImpl(ListServiceProviderRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return serviceProviderService.list(request, authSubject);
    }

    @Override
    public void create(CreateServiceProviderRequest request, StreamObserver<ServiceProvider> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> createImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ServiceProvider> createImpl(CreateServiceProviderRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return serviceProviderService.create(request, authSubject);
    }

    @Override
    public void update(UpdateServiceProviderRequest request, StreamObserver<ServiceProvider> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> updateImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<ServiceProvider> updateImpl(UpdateServiceProviderRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return serviceProviderService.update(request, authSubject);
    }

    @Override
    public void delete(DeleteServiceProviderRequest request, StreamObserver<Empty> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> deleteImpl(request))
                        .exceptionally(throwable -> {
                            throw GrpcApiExceptionResolver.doResolveException(throwable);
                        })
                , responseObserver);
    }

    private CompletableFuture<Empty> deleteImpl(DeleteServiceProviderRequest request) {
        AuthSubject authSubject = AuthenticationInterceptor.getAuthSubject(Context.current());
        return serviceProviderService.delete(request, authSubject);
    }
}
