package ru.yandex.solomon.gateway.api.v3alpha.priv;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.v3.priv.CreateServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.DeleteServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.GetServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderResponse;
import ru.yandex.monitoring.v3.priv.Reference;
import ru.yandex.monitoring.v3.priv.ServiceProvider;
import ru.yandex.monitoring.v3.priv.ShardSettings;
import ru.yandex.monitoring.v3.priv.ShardSettings.AggrRule.Function;
import ru.yandex.monitoring.v3.priv.UpdateServiceProviderRequest;
import ru.yandex.solomon.abc.validator.AbcServiceFieldValidator;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ServiceProviderService {
    private final InternalAuthorizer authorizer;
    private final AbcServiceFieldValidator abcServiceFieldValidator;
    private final ServiceProvidersDao serviceProvidersDao;
    private final boolean isCloud;

    @Autowired
    public ServiceProviderService(
            Optional<TGatewayCloudConfig> config,
            InternalAuthorizer authorizer,
            AbcServiceFieldValidator abcServiceFieldValidator,
            ServiceProvidersDao serviceProvidersDao)
    {
        this.authorizer = authorizer;
        this.abcServiceFieldValidator = abcServiceFieldValidator;
        this.serviceProvidersDao = serviceProvidersDao;
        this.isCloud = config.isPresent();
    }

    public CompletableFuture<ServiceProvider> get(GetServiceProviderRequest request, AuthSubject authSubject) {
        ServiceProviderValidator.validate(request);

        return serviceProvidersDao.read(request.getServiceProviderId()).thenApply(serviceProviderOpt -> {
            if (serviceProviderOpt.isEmpty()) {
                throw serviceProviderNotFound(request.getServiceProviderId());
            }
            return fromModel(serviceProviderOpt.get());
        });
    }

    public CompletableFuture<ListServiceProviderResponse> list(ListServiceProviderRequest request, AuthSubject authSubject) {
        ServiceProviderValidator.validate(request);

        return serviceProvidersDao.list(request.getFilter(), (int) request.getPageSize(), request.getPageToken())
                .thenApply(page -> {
                    TokenBasePage<ServiceProvider> mappedPage = page.map(ServiceProviderService::fromModel);
                    return ListServiceProviderResponse.newBuilder()
                            .addAllServiceProviders(mappedPage.getItems())
                            .setNextPageToken(mappedPage.getNextPageToken())
                            .build();
                });
    }

    public CompletableFuture<ServiceProvider> create(CreateServiceProviderRequest request, AuthSubject authSubject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());

        ServiceProviderValidator.validate(request, isCloud);

        return validateAbcService(request.getAbcService(), true).thenCompose(aVoid -> {
            String login = authSubject.getUniqueId();

            ServiceProvider serviceProvider = ServiceProvider.newBuilder()
                    .setId(request.getServiceProviderId())
                    .setDescription(request.getDescription())
                    .setShardSettings(request.getShardSettings())
                    .addAllReferences(request.getReferencesList())
                    .setAbcService(request.getAbcService())
                    .setCloudId(request.getCloudId())
                    .setTvmDestId(request.getTvmDestId())
                    .addAllIamServiceAccountIds(request.getIamServiceAccountIdsList())
                    .addAllTvmServiceIds(request.getTvmServiceIdsList())
                    .setIamServiceAccountId(request.getIamServiceAccountId())
                    .setHasGlobalId(request.getHasGlobalId())
                    .setCreatedAt(now)
                    .setUpdatedAt(now)
                    .setCreatedBy(login)
                    .setUpdatedBy(login)
                    .build();

            return authorizer.authorize(authSubject)
                    .thenCompose(account -> serviceProvidersDao.insert(toModel(serviceProvider)))
                    .thenApply(inserted -> {
                        if (!inserted) {
                            throw new ConflictException("service provider " + request.getServiceProviderId() + " already exists");
                        }
                        return serviceProvider;
                    });
        });
    }

    public CompletableFuture<ServiceProvider> update(UpdateServiceProviderRequest request, AuthSubject authSubject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());

        ServiceProviderValidator.validate(request, isCloud);

        return validateAbcService(request.getAbcService(), false).thenCompose(aVoid -> {
            ServiceProvider serviceProvider = ServiceProvider.newBuilder()
                    .setId(request.getServiceProviderId())
                    .setDescription(request.getDescription())
                    .setShardSettings(request.getShardSettings())
                    .addAllReferences(request.getReferencesList())
                    .setAbcService(request.getAbcService())
                    .setCloudId(request.getCloudId())
                    .setTvmDestId(request.getTvmDestId())
                    .setIamServiceAccountId(request.getIamServiceAccountId())
                    .addAllIamServiceAccountIds(request.getIamServiceAccountIdsList())
                    .addAllTvmServiceIds(request.getTvmServiceIdsList())
                    .setUpdatedAt(now)
                    .setHasGlobalId(request.getHasGlobalId())
                    .setUpdatedBy(authSubject.getUniqueId())
                    .setVersion(request.getVersion())
                    .build();

            return authorizer.authorize(authSubject)
                    .thenCompose(account -> serviceProvidersDao.update(toModel(serviceProvider)))
                    .thenCompose(updatedOpt -> {
                        //noinspection OptionalIsPresent
                        if (updatedOpt.isPresent()) {
                            return CompletableFuture.completedFuture(fromModel(updatedOpt.get()));
                        }

                        return serviceProvidersDao.exists(serviceProvider.getId())
                                .thenApply(exists -> {
                                    if (exists) {
                                        String message = String.format(
                                                "service provider \"%s\" with version %s is out of date",
                                                serviceProvider.getId(),
                                                serviceProvider.getVersion()
                                        );
                                        throw new ConflictException(message);
                                    }
                                    throw serviceProviderNotFound(serviceProvider.getId());
                                });
                    });
        });
    }

    public CompletableFuture<Empty> delete(DeleteServiceProviderRequest request, AuthSubject authSubject) {
        ServiceProviderValidator.validate(request);

        return authorizer.authorize(authSubject)
                .thenCompose(account -> serviceProvidersDao.delete(request.getServiceProviderId()))
                .thenApply(deleted -> {
                    if (!deleted) {
                        throw serviceProviderNotFound(request.getServiceProviderId());
                    }
                    return Empty.getDefaultInstance();
                });
    }


    private static ru.yandex.solomon.core.db.model.ServiceProvider toModel(ServiceProvider serviceProvider) {
        var iamServiceAccountIds = StringUtils.isEmpty(serviceProvider.getIamServiceAccountId())
                ? serviceProvider.getIamServiceAccountIdsList()
                : List.of(serviceProvider.getIamServiceAccountId());
        return ru.yandex.solomon.core.db.model.ServiceProvider.newBuilder()
                .setId(serviceProvider.getId())
                .setDescription(serviceProvider.getDescription())
                .setShardSettings(toModel(serviceProvider.getShardSettings()))
                .setReferences(toModel(serviceProvider.getReferencesList()))
                .setAbcService(serviceProvider.getAbcService())
                .setCloudId(serviceProvider.getCloudId())
                .setTvmDestId(serviceProvider.getTvmDestId())
                .setIamServiceAccountIds(iamServiceAccountIds)
                .setTvmServiceIds(new IntArrayList(serviceProvider.getTvmServiceIdsList()))
                .setCreatedAtMillis(Timestamps.toMillis(serviceProvider.getCreatedAt()))
                .setUpdatedAtMillis(Timestamps.toMillis(serviceProvider.getUpdatedAt()))
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setUpdatedBy(serviceProvider.getUpdatedBy())
                .setVersion((int) serviceProvider.getVersion())
                .setHasGlobalId(serviceProvider.getHasGlobalId())
                .build();
    }

    private static ServiceProvider fromModel(ru.yandex.solomon.core.db.model.ServiceProvider serviceProvider) {
        return ServiceProvider.newBuilder()
                .setId(serviceProvider.getId())
                .setDescription(serviceProvider.getDescription())
                .setHasGlobalId(serviceProvider.isHasGlobalId())
                .setShardSettings(fromModel(serviceProvider.getShardSettings()))
                .addAllReferences(fromModel(serviceProvider.getReferences()))
                .setAbcService(serviceProvider.getAbcService())
                .setCloudId(serviceProvider.getCloudId())
                .setTvmDestId(serviceProvider.getTvmDestId())
                .addAllIamServiceAccountIds(serviceProvider.getIamServiceAccountIds())
                .addAllTvmServiceIds(serviceProvider.getTvmServiceIds())
                .setIamServiceAccountId(Iterables.getFirst(serviceProvider.getIamServiceAccountIds(), ""))
                .setCreatedAt(Timestamps.fromMillis(serviceProvider.getCreatedAtMillis()))
                .setUpdatedAt(Timestamps.fromMillis(serviceProvider.getUpdatedAtMillis()))
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setUpdatedBy(serviceProvider.getUpdatedBy())
                .setVersion(serviceProvider.getVersion())
                .build();
    }

    private static ServiceProviderShardSettings toModel(ShardSettings shardSettings) {
        ServiceMetricConf.AggrRule[] aggrRules = shardSettings.getAggrRulesList().stream()
                .map(ServiceProviderService::toModel)
                .toArray(ServiceMetricConf.AggrRule[]::new);

        ServiceMetricConf metricConf = ServiceMetricConf.of(aggrRules, shardSettings.getMemOnly());

        return new ServiceProviderShardSettings(
                metricConf,
                (int) shardSettings.getMetricTtlDays(),
                (int) shardSettings.getGridSeconds(),
                (int) shardSettings.getIntervalSeconds());
    }

    private static ServiceMetricConf.AggrRule toModel(ShardSettings.AggrRule x) {
        var cond = x.getConditionsList().toArray(String[]::new);
        var targets = x.getTargetsList().toArray(String[]::new);
        var function = toModel(x.getFunction());
        return new ServiceMetricConf.AggrRule(cond, targets, function);
    }

    private static ShardSettings.AggrRule fromModel(ServiceMetricConf.AggrRule model) {
        var cond = Arrays.asList(model.getCond());
        var targets = Arrays.asList(model.getTarget());
        var function = fromModel(model.getAggr());
        return ShardSettings.AggrRule.newBuilder()
                .addAllConditions(cond)
                .addAllTargets(targets)
                .setFunction(function)
                .build();
    }

    private static MetricAggregation toModel(ShardSettings.AggrRule.Function function) {
        return switch (function) {
            case SUM -> MetricAggregation.SUM;
            case LAST -> MetricAggregation.LAST;
            default -> null;
        };
    }

    private static ShardSettings.AggrRule.Function fromModel(@Nullable MetricAggregation model) {
        if (model == null) {
            return Function.FUNCTION_UNSPECIFIED;
        }

        return switch (model) {
            case SUM -> Function.SUM;
            case LAST -> Function.LAST;
        };
    }

    private static ShardSettings fromModel(ServiceProviderShardSettings shardSettings) {
        var aggrRules = Arrays.stream(shardSettings.getMetricConf().getAggrRules())
                .map(ServiceProviderService::fromModel)
                .collect(Collectors.toList());

        return ShardSettings.newBuilder()
                .addAllAggrRules(aggrRules)
                .setMemOnly(shardSettings.getMetricConf().isRawDataMemOnly())
                .setMetricTtlDays(shardSettings.getMetricsTtlDays())
                .setGridSeconds(shardSettings.getGrid())
                .setIntervalSeconds(shardSettings.getInterval())
                .build();
    }

    private static List<ReferenceConf> toModel(List<Reference> references) {
        return references.stream()
                .map(ServiceProviderService::toModel)
                .collect(Collectors.toList());
    }

    private static ReferenceConf toModel(Reference reference) {
        return new ReferenceConf(
                reference.getLabel(),
                reference.getServicesList(),
                reference.getTypesList(),
                reference.getCrossFolder()
        );
    }

    private static List<Reference> fromModel(List<ReferenceConf> references) {
        return references.stream()
                .map(ServiceProviderService::fromModel)
                .collect(Collectors.toList());
    }

    private static Reference fromModel(ReferenceConf reference) {
        return Reference.newBuilder()
                .setLabel(reference.label)
                .addAllServices(reference.services)
                .addAllTypes(reference.types)
                .setCrossFolder(reference.crossFolder)
                .build();
    }

    private CompletableFuture<Void> validateAbcService(String abcService, boolean isNew) {
        if (StringUtils.isBlank(abcService)) {
            return CompletableFuture.completedFuture(null);
        }

        return abcServiceFieldValidator.validate(abcService, isNew);
    }

    private static NotFoundException serviceProviderNotFound(String id) {
        return new NotFoundException("no service provider with id " + id + " found");
    }
}
