package ru.yandex.solomon.gateway.api.v3alpha.priv;

import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.v3.priv.CreateServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.DeleteServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.GetServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.ListServiceProviderRequest;
import ru.yandex.monitoring.v3.priv.Reference;
import ru.yandex.monitoring.v3.priv.ShardSettings;
import ru.yandex.monitoring.v3.priv.UpdateServiceProviderRequest;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.solomon.core.conf.aggr.LabelValueExpr;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ServiceProviderValidator {
    private static final int MAX_AGGR_RULES_COUNT = 10;

    public static void validate(ListServiceProviderRequest request) {
    }

    public static void validate(GetServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
    }

    public static void validate(DeleteServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
    }

    public static void validate(CreateServiceProviderRequest request, boolean isCloud) {
        validateId(request.getServiceProviderId());
        IdValidator.ensureValid(request.getServiceProviderId(), "service provider");
        validateCloudFields(request, isCloud);
        validateShardSettings(request.getShardSettings());
        validateReferences(request.getReferencesList());
        // TODO(ivanzhukov): validate tvmServiceIds
    }

    public static void validate(UpdateServiceProviderRequest request, boolean isCloud) {
        validateId(request.getServiceProviderId());
        validateCloudFields(request, isCloud);
        validateShardSettings(request.getShardSettings());
        validateReferences(request.getReferencesList());
        if (request.getVersion() < 0) {
            throw new BadRequestException("version cannot be negative");
        }
    }

    private static void validateId(String serviceProviderId) {
        if (StringUtils.isBlank(serviceProviderId)) {
            throw new BadRequestException("service provider ID cannot be blank");
        }
    }

    private static void validateShardSettings(ShardSettings shardSettings) {
        validateAggrRules(shardSettings.getAggrRulesList());
        validateGrid(shardSettings.getGridSeconds());
    }

    private static void validateAggrRules(List<ShardSettings.AggrRule> aggrRules) {
        if (aggrRules.size() > MAX_AGGR_RULES_COUNT) {
            throw new BadRequestException("too many aggregation rules, max count is " + MAX_AGGR_RULES_COUNT);
        }

        for (ShardSettings.AggrRule rule : aggrRules) {
            List<String> cond = rule.getConditionsList();
            List<String> target = rule.getTargetsList();
            if (cond.isEmpty() && target.isEmpty()) {
                continue;
            }
            checkNonBlankStrings(cond, "aggrRule", "conditions");
            checkNonBlankStrings(target, "aggrRule", "targets");
            for (String c : cond) {
                checkLabel(c, "condition in aggregation rule");
            }
            for (String t : target) {
                checkLabel(t, "target in aggregation rule");
            }
        }
    }

    private static void checkNonBlankStrings(@Nonnull List<String> strs, String ruleType, String fieldName) {
        if (strs.isEmpty()) {
            throw new BadRequestException(fieldName + " in " + ruleType + " cannot be an empty");
        }
        for (String str : strs) {
            if (StringUtils.isBlank(str)) {
                throw new BadRequestException(fieldName + " in " + ruleType + " cannot have blank element");
            }
        }
    }

    private static void checkLabel(@Nonnull String labelStr, String message) {
        String[] nameValue = StringUtils.split(labelStr, '=');
        if (nameValue.length != 2) {
            throw new BadRequestException(message + " must be in 'name=value' format, but got: '" + labelStr + '\'');
        }
        if (!LabelValidator.isValidNameChars(nameValue[0])) {
            throw new BadRequestException("invalid label name for " + message + ", got: '" + nameValue[0] + '\'');
        }
        if (!LabelValueExpr.isValid(nameValue[1])) {
            throw new BadRequestException("invalid label value expression for " + message + ", got: '" + nameValue[1] + '\'');
        }
    }

    private static void validateGrid(long gridSeconds) {
        if (gridSeconds != Service.GRID_ABSENT && gridSeconds != Service.GRID_UNKNOWN) {
            if (gridSeconds < -1) {
                throw new BadRequestException("grid seconds can't be less than -1");
            }
            long decimGridSec = TimeUnit.MINUTES.toSeconds(5);
            if (!InstantUtils.checkGridMultiplicity(decimGridSec, gridSeconds)) {
                throw new BadRequestException("grid seconds (" + gridSeconds + ") must be multiple of 5m");
            }
        }
    }

    private static void validateReferences(List<Reference> references) {
        for (Reference reference : references) {
            validateReference(reference);
        }
    }

    private static void validateReference(Reference reference) {
        if (reference.getLabel().isEmpty()) {
            throw new BadRequestException("label in reference can't be empty");
        }
        if (!LabelsValidator.isKeyValid(reference.getLabel())) {
            throw new BadRequestException("label in reference is invalid");
        }
    }

    private static void validateCloudFields(CreateServiceProviderRequest request, boolean isCloud) {
        validateCloudFields(
                isCloud,
                request.getCloudId(),
                request.getAbcService(),
                request.getTvmDestId(),
                request.getIamServiceAccountId(),
                request.getIamServiceAccountIdsList());
    }

    private static void validateCloudFields(UpdateServiceProviderRequest request, boolean isCloud) {
        validateCloudFields(
                isCloud,
                request.getCloudId(),
                request.getAbcService(),
                request.getTvmDestId(),
                request.getIamServiceAccountId(),
                request.getIamServiceAccountIdsList());
    }

    private static void validateCloudFields(
        boolean isCloud,
        String cloudId,
        String abcService,
        String tvmDestId,
        String iamServiceAccountId,
        List<String> iamServiceAccountIds)
    {
        if (isCloud) {
            if (StringUtils.isBlank(cloudId)) {
                throw new BadRequestException("cloud ID must be not blank in cloud service provider");
            }
            if (StringUtils.isNoneBlank(abcService)) {
                throw new BadRequestException("ABC service must be blank in cloud service provider");
            }
            if (StringUtils.isNoneBlank(tvmDestId)) {
                throw new BadRequestException("TVM destination ID must be blank in cloud service provider");
            }
        } else {
            if (StringUtils.isBlank(abcService)) {
                throw new BadRequestException("ABC service must be not blank in cloud service provider");
            }
            if (StringUtils.isNoneBlank(cloudId)) {
                throw new BadRequestException("cloud ID must be blank in service provider");
            }
            if (StringUtils.isNoneBlank(iamServiceAccountId) || !iamServiceAccountIds.isEmpty()) {
                throw new BadRequestException("IAM service account ID must be blank in service provider");
            }
        }
    }
}
