package ru.yandex.solomon.gateway.backend.meta.search;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;
import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.solomon.gateway.backend.meta.FrontendMetric;
import ru.yandex.solomon.gateway.backend.meta.FrontendMetricPlainImpl;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.MetabaseClientException;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.time.Deadline;

/**
 * @author chebik
 */
@ThreadSafe
@ParametersAreNonnullByDefault
public class MetaSearch {

    @WillNotClose
    private final MetricsClient metricsClient;

    private MetaSearch(MetricsClient metricsClient) {
        this.metricsClient = metricsClient;
    }

    public CompletableFuture<ShardSearchResult> search(ShardKey shardKey, Selectors selectorsAfterShard, int limit, @Nullable String clusterId, Deadline deadline) {
        if (limit <= 0) {
            //limit == 0 will return only count of metrics, it's not supported by this search yet
            return CompletableFuture.failedFuture(
                new IllegalArgumentException("Only positive limit is supported")
            );
        }

        if (hasShardGlobOrInvalidSelectors(shardKey, selectorsAfterShard)) {
            return CompletableFuture.failedFuture(
                new RuntimeException("Invalid selectors passed: " + selectorsAfterShard + ". Shard selectors must be " + shardKey.toLabels())
            );
        }

        Selectors selectors = ShardSelectors.concat(shardKey, selectorsAfterShard);

        return metricsClient.find(FindRequest.newBuilder()
                .setLimit(limit)
                .setDeadline(deadline.toEpochMilli())
                .setSelectors(selectors)
                .setDestination(clusterId)
                .build())
                .thenApply(response -> {
                    int totalCount =
                        response.getMetricsCountByDestination().values()
                            .stream()
                            .mapToInt(Integer::intValue)
                            .max()
                            .orElse(0);

                    if (!response.isOk()) {
                        throw new MetabaseClientException(response.getStatus());
                    }

                    return new ShardSearchResult(
                            response.getMetrics()
                                    .stream()
                                    .map(this::metricFromProto)
                                    .collect(Collectors.toList()),
                            response.isTruncated()
                    );
                });
    }

    @Nullable
    private FrontendMetric metricFromProto(@Nullable MetricKey key) {
        if (key == null) {
            return null;
        }

        return new FrontendMetricPlainImpl(key);
    }

    private boolean hasShardGlobOrInvalidSelectors(ShardKey shardKey, Selectors selectorsAfterShard) {
        for (Selector selector : ShardSelectors.onlyShardKey(selectorsAfterShard)) {
            if ((selector.getType() != SelectorType.EXACT && selector.getType() != SelectorType.GLOB)) {
                return true;
            }

            switch (selector.getKey()) {
                case LabelKeys.PROJECT:
                    if (!shardKey.getProject().equals(selector.getValue())) {
                        return true;
                    }
                    break;
                case LabelKeys.CLUSTER:
                    if (!shardKey.getCluster().equals(selector.getValue())) {
                        return true;
                    }
                    break;
                case LabelKeys.SERVICE:
                    if (!shardKey.getService().equals(selector.getValue())) {
                        return true;
                    }
                    break;
            }
        }
        return false;
    }
}
