package ru.yandex.solomon.gateway.backend.meta.search;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class PageMetricStatsUtils {
    private PageMetricStatsUtils() {}

    public static List<String> build(Stream<Labels> labelsList) {
        Object2IntOpenHashMap<Label> metricsCountByLabel = new Object2IntOpenHashMap<>();
        MutableInt allMetricsCount = new MutableInt(0);

        labelsList.forEach(labels -> {
            labels.forEach(l -> metricsCountByLabel.addTo(l, 1));
            allMetricsCount.increment();
        });

        return buildImpl(allMetricsCount.intValue(), metricsCountByLabel);
    }

    private static List<String> buildImpl(
        int allMetricsCount,
        Object2IntOpenHashMap<Label> metricsCountByLabel)
    {
        Map<String, LabelStatsBuilder> labelNameToLabelStatsMap = new HashMap<>((int) (1.5 * metricsCountByLabel.size()));

        for (Object2IntMap.Entry<Label> entry : metricsCountByLabel.object2IntEntrySet()) {
            Label label = entry.getKey();
            LabelStatsBuilder labelStatsBuilder = labelNameToLabelStatsMap.computeIfAbsent(
                label.getKey(),
                key -> new LabelStatsBuilder(key, TextWithNumbersComparator.instance));
            labelStatsBuilder.countByValue.addTo(label.getValue(), entry.getIntValue());
        }

        Map<String, PageLabelStats> statsByLabel = new HashMap<>((int) (1.5 * labelNameToLabelStatsMap.size()));

        for (Map.Entry<String, LabelStatsBuilder> e : labelNameToLabelStatsMap.entrySet()) {
            statsByLabel.put(e.getKey(), e.getValue().build(allMetricsCount));
        }

        return statsByLabel.values().stream()
                .filter(e -> e.valueCountIncludingAbsent() > 1)
                .map(PageLabelStats::getLabelName)
                .collect(Collectors.toList());
    }

    private static class LabelStatsBuilder {
        final String name;
        final Comparator<String> comparator;

        Object2IntOpenHashMap<String> countByValue = new Object2IntOpenHashMap<>();

        LabelStatsBuilder(
            String name,
            @Nullable Comparator<String> comparator)
        {
            this.name = name;
            this.comparator = comparator;
        }

        PageLabelStats build(int allMetricCount) {
            HashMap<String, Long> map = new HashMap<>((int)(1.5 * countByValue.size()));

            for (Object2IntMap.Entry<String> e : countByValue.object2IntEntrySet()) {
                map.put(e.getKey(), (long) e.getIntValue());
            }

            return new PageLabelStats(name, map, allMetricCount);
        }
    }

    private static class PageLabelStats {
        private final String labelName;
        // WARNING: Do not expose values of metricCountByValue map, as they will be all equal to 1 in nearest future.
        // Keeping 1 as a value, however, doesn't violate the contract of this class
        private final Map<String, Long> metricCountByValue;
        private final boolean hasMetricsWithoutLabel;

        public PageLabelStats(String labelName, Map<String, Long> metricCountByValue, int allMetricCount) {
            this.labelName = labelName;
            this.metricCountByValue = metricCountByValue;
            int metricCountWithLabel = (int) metricCountByValue.values().stream().mapToLong(i -> i).sum();
            int metricCountWithoutLabel = allMetricCount - metricCountWithLabel;
            if (metricCountWithoutLabel < 0) {
                throw new RuntimeException("allMetricCount (" + allMetricCount + ") is less than metricCountWithLabel (" + metricCountWithLabel + ")");
            }
            this.hasMetricsWithoutLabel = metricCountWithoutLabel > 0;
        }

        public String getLabelName() {
            return labelName;
        }

        public int valueCountIncludingAbsent() {
            return metricCountByValue.size() + (hasMetricsWithoutLabel ? 1 : 0);
        }
    }
}
