package ru.yandex.solomon.gateway.backend.storage;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.stats.ArchiveStats;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.math.doubles.AggregateFunctionType;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.Int64Summary;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossDcGraphDataClient implements GraphDataClient {
    private final MetricsClient client;

    public CrossDcGraphDataClient(MetricsClient client) {
        this.client = client;
    }

    @Override
    public CompletableFuture<List<FetchResult>> fetch(
            List<MetricStorageRequest> requests,
            @Nullable String destination,
            Deadline deadline,
            String subjectId)
    {
        return CompletableFutures.safeCall(() -> requests.stream()
                .map(request -> fetch(request, destination, deadline, subjectId))
                .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(list -> list));
    }

    private CompletableFuture<FetchResult> fetch(
            MetricStorageRequest request,
            @Nullable String destination,
            Deadline deadline,
            String subjectId)
    {
        return client.readMany(convertRequest(request, destination, deadline, subjectId))
                .thenApply(response -> {
                    if (!response.isOk()) {
                        if (response.getStatus().getCode() == EStockpileStatusCode.METRIC_NOT_FOUND) {
                            return new FetchResult(AggrGraphDataArrayList.empty(), ArchiveStats.empty);
                        }

                        throw new StockpileClientException(response.getStatus());
                    }

                    Metric<MetricKey> metric = response.getMetrics().get(0);
                    AggrGraphDataArrayList list = metric.getTimeseries() != null
                            ? AggrGraphDataArrayList.of(metric.getTimeseries())
                            : AggrGraphDataArrayList.empty();

                    final ArchiveStats stats;

                    if (metric.getSummary() == null || metric.getSummary().getCount() == 0) {
                        stats = ArchiveStats.empty;
                    } else if (metric.getSummary() instanceof DoubleSummary) {
                        DoubleSummary summary = (DoubleSummary) metric.getSummary();
                        stats = new ArchiveStats(summary.getMin(), summary.getMax(), summary.getAvg(), summary.getLast(), summary.getSum());
                    } else if (metric.getSummary() instanceof Int64Summary) {
                        Int64Summary summary = (Int64Summary) metric.getSummary();
                        stats = new ArchiveStats(
                            (double) summary.getMin(),
                            (double) summary.getMax(),
                            (double) summary.getAvg(),
                            (double) summary.getLast(),
                            (double) summary.getSum());
                    } else {
                        stats = ArchiveStats.empty;
                    }

                    MetricType type = metric.getType();
                    if (!list.isEmpty() && type != MetricType.DGAUGE && !isHistogram(type)) {
                        list = AggrGraphDataArrayList.of(MetricTypeTransfers.of(type, MetricType.DGAUGE, list.iterator()));
                    }
                    return new FetchResult(list, stats);
                });
    }

    private ReadManyRequest convertRequest(
        MetricStorageRequest request,
        @Nullable String destination,
        Deadline deadline,
        String subjectId)
    {
        ReadManyRequest.Builder builder = ReadManyRequest.newBuilder();

        var type = request.getKey().getType();
        boolean isHistogram = isHistogram(type);

        AggregateFunctionType aggrType = isHistogram ? AggregateFunctionType.LAST : request.getAggrType();

        builder.addKey(request.getKey());
        builder.setInterval(request.getInterval());
        builder.setDestination(destination);
        builder.setDeadline(deadline.toEpochMilli());
        builder.setProducer(RequestProducer.STAFF);
        builder.setSubjectId(subjectId);

        if (request.getGridMillis() != 0) {
            builder.addOperation(Operation.newBuilder()
                    .setDownsampling(OperationDownsampling.newBuilder()
                            .setGridMillis(request.getGridMillis())
                            .setAggregation(convert(aggrType))
                            .setFillOption(request.getFillOption())
                            .setIgnoreMinStepMillis(request.isIgnoreMinStepMillis())
                            .build())
                    .build());
        }

        if (request.getRankFilter() != null) {
            builder.addOperation(Operation.newBuilder()
                    .setTop(request.getRankFilter())
                    .build());
        }

        return builder.build();
    }

    private boolean isHistogram(MetricType type) {
        return type == MetricType.HIST || type == MetricType.HIST_RATE;
    }

    private boolean isHistogram(ru.yandex.monlib.metrics.MetricType type) {
        return type == ru.yandex.monlib.metrics.MetricType.HIST || type == ru.yandex.monlib.metrics.MetricType.HIST_RATE;
    }

    private Aggregation convert(AggregateFunctionType aggrType) {
        switch (aggrType) {
            case AVG:
                return Aggregation.AVG;
            case COUNT:
                return Aggregation.COUNT;
            case MAX:
                return Aggregation.MAX;
            case MIN:
                return Aggregation.MIN;
            case SUM:
                return Aggregation.SUM;
            case LAST:
                return Aggregation.LAST;
            case DEFAULT:
                return Aggregation.DEFAULT_AGGREGATION;
            default:
                throw new UnsupportedOperationException("Unsupported type: " + aggrType);
        }
    }
}
