package ru.yandex.solomon.gateway.backend.storage;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.gateway.backend.data.MetricDataRequest;
import ru.yandex.solomon.gateway.backend.data.MetricWithGraphData;
import ru.yandex.solomon.math.doubles.AggregateFunctionType;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.Deadline;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import({
    GraphDataFetchContext.class,
    ReadMetrics.class
})
@ParametersAreNonnullByDefault
public class FrontendGraphDataLoader {

    private final GraphDataClient kikimrProvider;
    private final ReadMetrics metrics;

    public FrontendGraphDataLoader(
        GraphDataClient kikimrProvider,
        ReadMetrics metrics)
    {
        this.kikimrProvider = kikimrProvider;
        this.metrics = metrics;
    }

    CompletableFuture<List<MetricWithGraphData>> readMetricsFromStorage(
        String shardId,
        String client,
        Collection<MetricDataRequest> metrics,
        long gridMillis,
        AggregateFunctionType aggrType,
        OperationDownsampling.FillOption fillOption,
        boolean ignoreMinStepMillis,
        @Nullable String destination,
        Deadline deadline,
        String subjectId)
    {
        if (metrics.size() > DataLimits.MAX_METRICS_FOR_AGGR_COUNT) {
            String message =
                "more than " + DataLimits.MAX_METRICS_FOR_AGGR_COUNT + " metrics requested: "
                    + metrics.size();

            throw new RuntimeException(message);
        }

        ReadMetrics.ShardMetrics shardMetrics = this.metrics.getClientMetrics(client)
            .getShardMetrics(shardId);

        List<MetricLoadState> metricLoadStates = metrics.stream()
            .peek(request -> shardMetrics.addReadInterval(request.getInterval()))
            .map(MetricLoadState::new)
            .collect(Collectors.toList());

        long startNanoTime = shardMetrics.started(metricLoadStates.size());
        CompletableFuture<Void> loadFromStorageFuture =
            loadFromStorage(metricLoadStates, gridMillis, aggrType, fillOption, ignoreMinStepMillis, destination, deadline, subjectId)
                .whenComplete((unit, throwable) -> shardMetrics.completed(metricLoadStates.size(), startNanoTime));

        return loadFromStorageFuture.thenApply(ignore -> metricLoadStates.stream()
            .map(MetricLoadState::finish)
            .collect(Collectors.toList()));
    }

    @Nonnull
    private CompletableFuture<Void> loadFromStorage(
        List<MetricLoadState> metricLoadStates,
        long gridMillis,
        AggregateFunctionType aggrType,
        OperationDownsampling.FillOption fillOption,
        boolean ignoreMinStepMillis,
        @Nullable String destination,
        Deadline deadline,
        String subjectId)
    {
        List<MetricStorageRequest> requests = metricLoadStates.stream()
            .map(s -> new MetricStorageRequest(
                s.request.getInterval(),
                s.request.getMetric().getKey(),
                gridMillis,
                aggrType,
                fillOption,
                ignoreMinStepMillis,
                null))
            .collect(Collectors.toList());

        CompletableFuture<List<FetchResult>> fetchFuture =
            kikimrProvider.fetch(
                requests,
                destination,
                deadline,
                subjectId);

        return fetchFuture.thenApply(readResponses -> {
            if (readResponses.size() != metricLoadStates.size()) {
                throw new IllegalStateException("Got " +
                    readResponses.size() + " responses for " +
                    metricLoadStates.size() + " requests");
            }

            for (int i = 0; i < readResponses.size(); i++) {
                FetchResult t = readResponses.get(i);
                MetricLoadState metricLoadState = metricLoadStates.get(i);

                metricLoadState.graphData = t.graphData;
            }
            return null;
        });
    }

    static class MetricLoadState {
        private final MetricDataRequest request;

        private boolean done = false;
        AggrGraphDataArrayList graphData;

        MetricLoadState(MetricDataRequest request) {
            this.request = request;
        }

        MetricWithGraphData finish() {
            if (done) {
                throw new IllegalStateException("failed to finish metric load state");
            }

            done = true;

            return new MetricWithGraphData(
                request,
                graphData
            );
        }
    }
}
