package ru.yandex.solomon.gateway.backend.storage;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiGet;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiMetric;
import ru.yandex.solomon.gateway.backend.client.data.WwwMetricValue;
import ru.yandex.solomon.gateway.backend.client.www.page.DownSamplingParams;
import ru.yandex.solomon.gateway.backend.data.MetricDataRequest;
import ru.yandex.solomon.gateway.backend.data.MetricWithGraphData;
import ru.yandex.solomon.gateway.backend.meta.search.MetaSearch;
import ru.yandex.solomon.gateway.backend.page.WwwQueryArgsUtils;
import ru.yandex.solomon.gateway.utils.UserLinks;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListViewIterator;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;


/**
 * @author Stepan Koltsov
 */
@Component
@Import(FrontendGraphDataLoader.class)
public class LocalDataApiManager {

    private final FrontendGraphDataLoader frontendGraphDataLoader;
    private final MetaSearch metaSearch;

    public LocalDataApiManager(
            FrontendGraphDataLoader frontendGraphDataLoader,
            MetaSearch metaSearch)
    {
        this.frontendGraphDataLoader = frontendGraphDataLoader;
        this.metaSearch = metaSearch;
    }

    public CompletableFuture<WwwDataApiGet> processGet(
        ShardConfDetailed shardConf,
        Interval interval,
        DownSamplingParams downsamplingParams,
        Map<String, String> params,
        @Nullable String clusterId,
        Deadline deadline,
        String subjectId)
    {
        ShardKey shardKey = shardConf.shardKey();

        LabelSelectorSet selectors = LabelSelectorSet.fromMap(UserLinks.cleanParams(params));
        return metaSearch.search(shardKey, selectors.toNewSelectors(), DataLimits.MAX_METRICS_COUNT, clusterId, deadline).thenApply(
            searchResult -> {
                if (searchResult.isTruncated()) {
                    throw new BadRequestException("Too many metrics requested. Maximum allowed is " + searchResult.size() +
                        " and more metrics than that were found");
                }

                return searchResult.getMetricsStream()
                    .map(s -> new MetricDataRequest(s, interval))
                    .collect(Collectors.toList());
            }
        ).thenCompose(
            requests -> {
                return frontendGraphDataLoader.readMetricsFromStorage(
                    shardConf.getId(),
                    "data-api",
                    requests,
                    downsamplingParams.getGridMillis(),
                    downsamplingParams.getAggrType(),
                    downsamplingParams.getFillOption(),
                    downsamplingParams.isIgnoreMinStepMillis(),
                    clusterId,
                    deadline,
                    subjectId);
            }
        ).thenApply(datas -> {
            List<WwwDataApiMetric> wwwMetrics = datas.parallelStream()
                    .map(LocalDataApiManager::newApiMetric)
                    .collect(Collectors.toList());
            return new WwwDataApiGet(wwwMetrics);
        });
    }

    private static WwwDataApiMetric newApiMetric(MetricWithGraphData s) {
        Labels labels = s.getLabels();

        AggrGraphDataArrayList source = s.getGraphData();
        AggrPoint point = new AggrPoint();
        AggrGraphDataArrayListViewIterator it = source.iterator();
        final List<WwwMetricValue> values = new ArrayList<>(source.length());
        while (it.next(point)) {
            double value = point.getValueDivided();
            if (Double.isNaN(value)) {
                continue;
            }

            values.add(new WwwMetricValue(Instant.ofEpochMilli(point.tsMillis).toString(), value));
        }
        String created = Instant.ofEpochSecond(s.createdSeconds()).toString();
        boolean deriv = s.isDeriv();

        return new WwwDataApiMetric(labels, created, deriv, values);
    }

    public CompletableFuture<List<WwwDataApiMetric>> findMetrics(
        ShardKey shardKey,
        Map<String, String> params,
        Deadline deadline)
    {
        LabelSelectorSet selectors = WwwQueryArgsUtils.selectorsFromQuery(params);

        return metaSearch.search(shardKey, selectors.toNewSelectors(), DataLimits.MAX_METRICS_FOR_AGGR_COUNT, null, deadline)
            .thenApply(searchResult -> searchResult.getMetricsStream()
                .map(metric -> {
                    Labels labels = metric.getLabels();
                    int createdSeconds = metric.getCreatedSeconds();
                    boolean deriv = (metric.getType() == MetricType.RATE);
                    String createdSecondsStr = Instant.ofEpochSecond(createdSeconds).toString();
                    return new WwwDataApiMetric(labels, createdSecondsStr, deriv, null);
                })
                .collect(Collectors.toList()));
    }
}
